# ✅ Sistema de Publicações - Implementação Completa

## 📋 Resumo

Sistema completo de **Publicações** implementado com funcionalidade de **alcance granular** para controlar onde as publicações serão exibidas.

---

## 🎯 Conceito

- **Criativo**: Arquivo de mídia (imagem/vídeo) reutilizável, aprovado pela matriz
- **Publicação**: Contrato de exibição de um criativo específico, com período definido e alcance configurável

---

## ✨ Funcionalidades Implementadas

### 1. CRUD Completo de Publicações
- ✅ Listagem com filtros avançados (status, franquia, cliente, período, busca)
- ✅ Cards de estatísticas (ativas, vigentes, pausadas, valor total)
- ✅ Criação de publicações
- ✅ Edição de publicações
- ✅ Exclusão de publicações
- ✅ Toggle de status (ativar/pausar)

### 2. Sistema de Alcance Granular ⭐ NOVO
Permite definir para quais equipamentos a publicação será exibida:

#### Opções de Alcance:
1. **Todos os Equipamentos** 
   - Badge: 🔵 Azul (primary)
   - Ícone: `ph-broadcast`
   - Exibe em todos os equipamentos do cliente

2. **Serial Específico**
   - Badge: 🟡 Amarelo (warning)
   - Ícone: `ph-device-mobile`
   - Exibe apenas no equipamento com o serial especificado
   - Campo obrigatório: `equipment_serial`

3. **Grupo de Equipamentos**
   - Badge: 🔵 Azul claro (info)
   - Ícone: `ph-folder-user`
   - Exibe apenas nos equipamentos do grupo especificado
   - Campo obrigatório: `grupo_equipamentos_id`
   - ⚠️ Sistema de grupos será implementado futuramente

### 3. Validações Implementadas

#### Validações de Negócio:
- ✅ Criativo deve estar aprovado pela matriz
- ✅ Franquia deve ser do tipo 'franquia'
- ✅ Cliente deve ser do tipo 'cliente'
- ✅ Data de início não pode estar no passado

#### Validações de Alcance (Condicionais):
```php
'tipo_alcance' => 'required|in:todos,serial_especifico,grupo',
'equipment_serial' => 'nullable|string|max:255|required_if:tipo_alcance,serial_especifico',
'grupo_equipamentos_id' => 'nullable|integer|required_if:tipo_alcance,grupo',
```

### 4. Interface Inteligente

#### UX Condicional (JavaScript):
- Campos aparecem/desaparecem conforme seleção do `tipo_alcance`
- Se escolher "Serial Específico" → mostra campo `equipment_serial`
- Se escolher "Grupo" → mostra campo `grupo_equipamentos_id`
- Se escolher "Todos" → oculta ambos os campos

#### Listagem Aprimorada:
- ✅ Coluna "Alcance" com badges coloridos
- ✅ Exibe serial/grupo quando aplicável
- ✅ Preview de criativo com thumbnail
- ✅ Informações de período e duração
- ✅ Badge "Vigente" para publicações ativas no período

### 5. Comando de Expiração Automática

```bash
# Executar manualmente (modo dry-run, apenas visualiza)
php artisan publicacoes:expire --dry-run

# Executar e atualizar status
php artisan publicacoes:expire
```

**Agendamento:**
- Execução automática: **diariamente às 00:01**
- Busca publicações ativas com `data_fim < hoje`
- Atualiza status para 'expirada'
- Exibe tabela com detalhes antes de confirmar

---

## 📊 Estrutura do Banco de Dados

### Tabela: `publicacoes`

| Campo | Tipo | Descrição |
|-------|------|-----------|
| `id` | BIGINT | ID único |
| `criativo_id` | BIGINT | FK para criativos (midias) |
| `franquia_id` | BIGINT | FK para empresas (tipo 'franquia') |
| `cliente_id` | BIGINT | FK para empresas (tipo 'cliente') |
| `titulo` | VARCHAR | Título da publicação |
| `data_inicio` | DATE | Início da exibição |
| `data_fim` | DATE | Fim da exibição |
| `valor_contrato` | DECIMAL | Valor do contrato (opcional) |
| `impressoes_contratadas` | INT | Qtd de impressões (opcional) |
| `status` | ENUM | ativa, pausada, expirada, cancelada |
| **`tipo_alcance`** ⭐ | ENUM | todos, serial_especifico, grupo |
| **`equipment_serial`** ⭐ | VARCHAR | Serial do equipamento (condicional) |
| **`grupo_equipamentos_id`** ⭐ | BIGINT | ID do grupo (condicional) |
| `publicada_por` | BIGINT | FK para users |
| `publicada_em` | TIMESTAMP | Data/hora da publicação |

---

## 🔗 Rotas Disponíveis

| Método | URI | Nome | Ação |
|--------|-----|------|------|
| GET | `/publicacoes` | publicacoes.index | Listagem |
| POST | `/publicacoes` | publicacoes.store | Criar |
| GET | `/publicacoes/create` | publicacoes.create | Formulário criação |
| GET | `/publicacoes/{id}/edit` | publicacoes.edit | Formulário edição |
| PUT/PATCH | `/publicacoes/{id}` | publicacoes.update | Atualizar |
| DELETE | `/publicacoes/{id}` | publicacoes.destroy | Deletar |
| PATCH | `/publicacoes/{id}/status` | publicacoes.toggleStatus | Ativar/Pausar |

---

## 🧪 Como Testar

### 1. Acessar a Listagem
```
http://localhost:8082/publicacoes
```

### 2. Criar Nova Publicação
1. Clique em "Nova Publicação"
2. Preencha os campos:
   - **Criativo**: Selecione um criativo aprovado
   - **Título**: Nome da publicação
   - **Franquia**: Selecione a franquia responsável
   - **Cliente**: Selecione o cliente anunciante
   - **Alcance**: Escolha o tipo:
     - ✅ **Todos**: Deixa outros campos ocultos
     - ✅ **Serial Específico**: Campo `equipment_serial` aparece automaticamente
     - ✅ **Grupo**: Campo `grupo_equipamentos_id` aparece automaticamente
   - **Período**: Data início e fim
   - **Valores**: Contrato e impressões (opcionais)
   - **Status**: ativa, pausada, etc.
3. Clique em "Salvar"

### 3. Testar Validação Condicional
1. Escolha "Serial Específico"
2. **NÃO** preencha o campo de serial
3. Tente salvar → Erro: "Campo equipment_serial é obrigatório quando tipo_alcance for serial_especifico"

### 4. Testar Toggle de Status
1. Na listagem, clique no botão de "Pausar" (⏸️) em uma publicação ativa
2. Verifique se o status mudou para "Pausada"
3. Clique em "Ativar" (▶️)
4. Verifique se voltou para "Ativa"

### 5. Testar Filtros
1. Use o campo de busca para filtrar por título
2. Selecione um status específico
3. Selecione uma franquia
4. Selecione um cliente
5. Clique em "Filtrar"

### 6. Testar Comando de Expiração
```bash
# Modo visualização (não altera dados)
php artisan publicacoes:expire --dry-run

# Executar de verdade
php artisan publicacoes:expire
```

---

## 🎨 Badges e Ícones

### Status:
- ✅ **Ativa**: Badge verde (`bg-success`)
- ⏸️ **Pausada**: Badge amarelo (`bg-warning`)
- ❌ **Expirada**: Badge vermelho (`bg-danger`)
- 🚫 **Cancelada**: Badge cinza (`bg-secondary`)

### Alcance:
- 📡 **Todos**: Badge azul (`bg-primary`) + ícone `ph-broadcast`
- 📱 **Serial Específico**: Badge amarelo (`bg-warning`) + ícone `ph-device-mobile`
- 👥 **Grupo**: Badge azul claro (`bg-info`) + ícone `ph-folder-user`

---

## 📝 Model - Métodos Úteis

### Scopes:
```php
Publicacao::ativas()->get();              // Apenas ativas
Publicacao::vigentes()->get();            // No período atual
Publicacao::expiradas()->get();           // Expiradas
Publicacao::pausadas()->get();            // Pausadas
Publicacao::paraTodos()->get();           // Alcance: todos
Publicacao::serialEspecifico('ABC123')->get(); // Serial específico
Publicacao::grupoEquipamentos(1)->get();  // Grupo específico
```

### Helpers:
```php
$publicacao->isVigente();                 // bool: está no período?
$publicacao->isExpirada();                // bool: já expirou?
$publicacao->diasRestantes();             // int: dias até expirar
$publicacao->duracaoTotal();              // int: duração total em dias
$publicacao->getTipoAlcanceLabel();       // string: label humanizado
$publicacao->aplicavelParaEquipamento($serial, $grupoId); // bool: aplica?
```

---

## 🚀 Próximos Passos

### 1. Sistema de Grupos de Equipamentos (Futuro)
- Criar tabela `grupos_equipamentos`
- Relacionamento: `Empresa hasMany GrupoEquipamentos`
- Relacionamento: `GrupoEquipamentos hasMany Hotspots`
- Interface para gerenciar grupos
- Popular select de grupos na view de publicações

### 2. API para Equipamentos (Futuro)
```php
// GET /api/publicacoes/vigentes?serial={serial}&grupo_id={id}
// Retorna apenas publicações aplicáveis ao equipamento
```

### 3. Dashboard de Métricas (Futuro)
- Publicações ativas por franquia
- Receita total (soma de valor_contrato)
- Publicações expirando em 7 dias
- Gráfico de evolução

### 4. Melhorias de UX (Futuro)
- Preview do criativo em modal ao clicar na thumbnail
- Calendário visual para seleção de período
- Autocomplete para serial de equipamento
- Relatório de performance de publicações

---

## 📂 Arquivos Modificados/Criados

### Models:
- ✅ `app/Models/Publicacao.php` (completo com scopes e helpers)

### Controllers:
- ✅ `app/Http/Controllers/PublicacaoController.php` (7 métodos)

### Views:
- ✅ `resources/views/publicacoes/index.blade.php` (listagem)
- ✅ `resources/views/publicacoes/create.blade.php` (formulário criação)
- ✅ `resources/views/publicacoes/edit.blade.php` (formulário edição)
- ✅ `resources/views/layouts/sidebar.blade.php` (menu adicionado)

### Migrations:
- ✅ `database/migrations/2025_10_17_092655_create_publicacoes_table.php`
- ✅ `database/migrations/2025_10_17_180512_add_equipment_fields_to_publicacoes_table.php`

### Commands:
- ✅ `app/Console/Commands/ExpirePublicacoes.php`

### Routes:
- ✅ `routes/web.php` (8 rotas adicionadas)
- ✅ `routes/console.php` (schedule adicionado)

---

## ✅ Checklist de Implementação

- [x] Model Publicacao com relacionamentos
- [x] 9 Scopes (ativas, vigentes, expiradas, etc.)
- [x] 6 Helper methods
- [x] PublicacaoController com CRUD completo
- [x] Validações de negócio
- [x] Validação condicional (required_if)
- [x] 8 Rotas (resource + toggleStatus)
- [x] View index com filtros e cards de estatísticas
- [x] View create com JavaScript condicional
- [x] View edit com JavaScript condicional
- [x] Campo tipo_alcance (ENUM)
- [x] Campo equipment_serial (condicional)
- [x] Campo grupo_equipamentos_id (condicional)
- [x] Badges visuais para alcance
- [x] Menu na sidebar
- [x] Comando ExpirePublicacoes
- [x] Schedule diário (00:01)
- [x] Dados de teste (3 criativos)

---

## 🎉 Status: 100% Completo

O sistema de publicações está **completamente funcional** e pronto para uso! 

Acesse: **http://localhost:8082/publicacoes**

---

**Data de Implementação**: 17/10/2025  
**Versão Laravel**: 12.0  
**PHP**: 8.2.29
