<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Models\PublicacaoImpressao;
use App\Models\Hotspot;
use App\Models\Criativo;
use App\Models\Publicacao;
use Carbon\Carbon;

class ProcessSplashAudits extends Command
{
    protected $signature = 'etl:process-splash-audits';
    protected $description = 'ETL process to transform splash audit data and load it into aggregated analytics tables.';

    public function handle()
    {
        $this->info('Starting Splash Audit ETL process...');
        Log::info('Starting Splash Audit ETL process...');

        $processedCount = 0;

        DB::transaction(function () use (&$processedCount) {
            $unprocessedLogs = DB::table('auditoria_splash')->where('processed', false)->get();
            
            if ($unprocessedLogs->isEmpty()) {
                $this->info('No new audit logs to process.');
                return;
            }

            $this->info($unprocessedLogs->count() . ' new audit logs found. Processing...');

            // Caching for performance
            $hotspots = Hotspot::all()->keyBy('serial');
            $criativos = Criativo::all()->keyBy('path');
            $publicacoes = Publicacao::with('criativo')->whereIn('criativo_id', $criativos->pluck('id'))->get()->groupBy('criativo_id');

            $aggregates = [];
            $processedIds = [];

            foreach ($unprocessedLogs as $log) {
                if (empty($log->imagem) || empty($log->hs_serial)) {
                    continue;
                }

                $criativo = $criativos->get($log->imagem);
                if (!$criativo) {
                    continue; // Criativo not found for this path
                }

                $hotspot = $hotspots->get($log->hs_serial);
                if (!$hotspot) {
                    continue; // Hotspot not found for this serial
                }

                $logTimestamp = Carbon::parse($log->created_at);
                $activePublication = null;

                if (isset($publicacoes[$criativo->id])) {
                    foreach ($publicacoes[$criativo->id] as $pub) {
                        if ($logTimestamp->between($pub->data_inicio, $pub->data_fim)) {
                            $activePublication = $pub;
                            break;
                        }
                    }
                }

                if ($activePublication) {
                    $logDate = $logTimestamp->toDateString();
                    $key = $logDate . '-' . $activePublication->id . '-' . $hotspot->id;

                    if (!isset($aggregates[$key])) {
                        $aggregates[$key] = [
                            'data' => $logDate,
                            'publicacao_id' => $activePublication->id,
                            'criativo_id' => $criativo->id,
                            'hotspot_id' => $hotspot->id,
                            'impressoes' => 0,
                            'usuarios_unicos' => [],
                        ];
                    }

                    $aggregates[$key]['impressoes']++;
                    if (!empty($log->cl_mac_address)) {
                        $aggregates[$key]['usuarios_unicos'][$log->cl_mac_address] = true;
                    }
                    $processedIds[] = $log->id;
                }
            }

            if (!empty($aggregates)) {
                foreach ($aggregates as $aggregate) {
                    $uniqueUsersCount = count($aggregate['usuarios_unicos']);
                    $impressao = PublicacaoImpressao::firstOrNew(
                        [
                            'data' => $aggregate['data'],
                            'publicacao_id' => $aggregate['publicacao_id'],
                            'hotspot_id' => $aggregate['hotspot_id'],
                            'criativo_id' => $aggregate['criativo_id'],
                        ]
                    );

                    $impressao->impressoes = ($impressao->impressoes ?? 0) + $aggregate['impressoes'];
                    // This logic for unique users is not perfect as it doesn't account for previous days, but it's a start.
                    // A better approach would be a separate daily unique user tracking table.
                    $impressao->usuarios_unicos = ($impressao->usuarios_unicos ?? 0) + $uniqueUsersCount;
                    $impressao->save();
                }
            }

            if (!empty($processedIds)) {
                DB::table('auditoria_splash')->whereIn('id', $processedIds)->update(['processed' => true]);
                $processedCount = count($processedIds);
            }
        });

        $this->info("ETL process finished. Processed {$processedCount} logs.");
        Log::info("ETL process finished. Processed {$processedCount} logs.");
    }
}
