<?php

namespace App\Http\Controllers;

use App\Models\Hotspot;
use App\Models\Empresa;
use Illuminate\Http\Request;

class HotspotController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Hotspot::with(['franquia', 'cliente']);

        // Filtro avançado por tipo
        if ($request->filled('filter_type') && $request->filled('search')) {
            $filterType = $request->input('filter_type');
            $searchTerm = $request->input('search');

            switch ($filterType) {
                case 'franquia':
                    $query->whereHas('franquia', function($q) use ($searchTerm) {
                        $q->where('name', 'like', '%' . $searchTerm . '%');
                    });
                    break;
                case 'cliente':
                    $query->whereHas('cliente', function($q) use ($searchTerm) {
                        $q->where('name', 'like', '%' . $searchTerm . '%');
                    });
                    break;
                case 'marca':
                    $query->where('marca', 'like', '%' . $searchTerm . '%');
                    break;
                case 'modelo':
                    $query->where('modelo', 'like', '%' . $searchTerm . '%');
                    break;
                case 'serial':
                    $query->where('serial', 'like', '%' . $searchTerm . '%');
                    break;
            }
        }

        $hotspots = $query->latest()->paginate(15)->withQueryString();

        return view('hotspots.index', compact('hotspots'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $empresas = Empresa::orderBy('name')->get();
        return view('hotspots.create', compact('empresas'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'marca' => 'required|string|max:255',
            'modelo' => 'required|string|max:255',
            'serial' => 'required|string|max:255|unique:hotspots',
            'franquia_id' => 'nullable|exists:empresas,id',
            'cliente_id' => 'nullable|exists:empresas,id',
            'situacao' => 'nullable|in:estoque,em_transito,aguardando_instalacao,instalado,manutencao,desativado',
            'vpn_ip' => 'nullable|ip',
            'vpn_login' => 'nullable|string|max:100',
            'vpn_password' => 'nullable|string|max:255',
        ]);

        Hotspot::create($request->all());

        return redirect()->route('hotspots.index')->with('success', 'Hotspot cadastrado com sucesso!');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Hotspot $hotspot)
    {
        $empresas = Empresa::orderBy('name')->get();
        return view('hotspots.edit', compact('hotspot', 'empresas'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Hotspot $hotspot)
    {
        $request->validate([
            'marca' => 'required|string|max:255',
            'modelo' => 'required|string|max:255',
            'serial' => 'required|string|max:255|unique:hotspots,serial,' . $hotspot->id,
            'franquia_id' => 'nullable|exists:empresas,id',
            'cliente_id' => 'nullable|exists:empresas,id',
            'situacao' => 'nullable|in:estoque,em_transito,aguardando_instalacao,instalado,manutencao,desativado',
            'vpn_ip' => 'nullable|ip',
            'vpn_login' => 'nullable|string|max:100',
            'vpn_password' => 'nullable|string|max:255',
        ]);

        $hotspot->update($request->all());

        return redirect()->route('hotspots.index')->with('success', 'Hotspot atualizado com sucesso!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Hotspot $hotspot)
    {
        $hotspot->delete();

        return redirect()->route('hotspots.index')->with('success', 'Hotspot excluído com sucesso!');
    }

    /**
     * Toggle status of the hotspot
     */
    public function toggleStatus(Hotspot $hotspot)
    {
        $hotspot->status = !$hotspot->status;
        $hotspot->save();

        return redirect()->route('hotspots.index')->with('success', 'Status alterado com sucesso!');
    }

    /**
     * Update situacao of the hotspot
     */
    public function updateSituacao(Request $request, Hotspot $hotspot)
    {
        $request->validate([
            'situacao' => 'required|in:estoque,em_transito,aguardando_instalacao,instalado,manutencao,desativado',
        ]);

        $hotspot->situacao = $request->situacao;
        $hotspot->save();

        return response()->json([
            'success' => true,
            'message' => 'Situação atualizada com sucesso!',
            'situacao' => $hotspot->situacao
        ]);
    }

    /**
     * Get trashed hotspots
     */
    public function trash()
    {
        $hotspots = Hotspot::onlyTrashed()
                        ->with('destroyer:id,name', 'empresa:id,name')
                        ->latest('deleted_at')
                        ->get();
        return response()->json($hotspots);
    }

    /**
     * Restore trashed hotspot
     */
    public function restore($id)
    {
        $hotspot = Hotspot::onlyTrashed()->findOrFail($id);
        $hotspot->restore();

        return redirect()->route('hotspots.index')->with('success', 'Hotspot restaurado com sucesso!');
    }
}
