<?php

namespace App\Http\Controllers;

use App\Models\Midia;
use App\Models\Empresa;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Intervention\Image\ImageManager;
use Intervention\Image\Drivers\Gd\Driver;

class MidiaController extends Controller
{
    public function index(Request $request)
    {
        $query = Midia::query();

        if ($request->filled('q')) {
            $query->where('title', 'like', '%' . $request->input('q') . '%');
        }

        $midias = $query->with('empresa')->latest()->paginate(15);

        return view('midias.index', compact('midias'));
    }

    public function create()
    {
        $empresas = Empresa::all();
        return view('midias.create', compact('empresas'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'tipo' => 'nullable|in:imagem,video,html,quiz,nps',  // Renomeado de 'type'
            'type' => 'nullable|in:image,video',  // Mantido para compatibilidade
            'media_file' => 'nullable|file',
            'empresa_legado_id' => 'required|exists:empresas,id',  // Renomeado de 'empresa_id'
            'equipment_serial' => 'nullable|string|max:255',
            'position' => 'nullable|integer|between:1,6',
            'start_datetime' => 'nullable|date',
            'end_datetime' => 'nullable|date|after:start_datetime',
        ]);

        $path = null;
        $tipo = $request->tipo ?? ($request->type === 'image' ? 'imagem' : 'video');
        $formato = 'jpg'; // default
        
        if ($request->hasFile('media_file')) {
            $extension = $request->file('media_file')->extension();
            $formato = in_array($extension, ['jpg', 'png', 'gif', 'mp4', 'html']) ? $extension : 'jpg';
            
            // Gerar nome criptografado: MD5(uniqid + timestamp + random)
            $uniqueString = uniqid('', true) . microtime(true) . rand(1000, 9999);
            $hashedName = md5($uniqueString);
            $filename = $hashedName . '.' . $extension;
            
            if ($request->type === 'image' || $tipo === 'imagem') {
                $manager = new ImageManager(new Driver());
                $image = $manager->read($request->file('media_file'));
                $image->resize(640, 1100);
                $path = 'midias/' . $filename;
                Storage::disk('public')->put($path, (string) $image->encode());
            } else {
                $path = 'midias/' . $filename;
                Storage::disk('public')->putFileAs('midias', $request->file('media_file'), $filename);
            }
        }

        Midia::create([
            'title' => $request->title,
            'tipo' => $tipo,
            'formato' => $formato,
            'path' => $path,
            'empresa_legado_id' => $request->empresa_legado_id,
            'equipment_serial' => $request->equipment_serial,
            'position' => $request->position,
            'start_datetime' => $request->start_datetime,
            'end_datetime' => $request->end_datetime,
            'status_aprovacao' => 'aprovado',  // Auto-aprovado por enquanto
        ]);

        return redirect()->route('midias.index');
    }

    public function edit(Midia $midia)
    {
        $empresas = Empresa::all();
        return view('midias.edit', compact('midia', 'empresas'));
    }

    public function update(Request $request, Midia $midia)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'tipo' => 'nullable|in:imagem,video,html,quiz,nps',  // Renomeado de 'type'
            'type' => 'nullable|in:image,video',  // Mantido para compatibilidade
            'media_file' => 'nullable|file',
            'empresa_legado_id' => 'required|exists:empresas,id',  // Renomeado de 'empresa_id'
            'equipment_serial' => 'nullable|string|max:255',
            'position' => 'nullable|integer|between:1,6',
            'start_datetime' => 'nullable|date',
            'end_datetime' => 'nullable|date|after:start_datetime',
        ]);

        $path = $midia->path;
        $tipo = $request->tipo ?? ($request->type === 'image' ? 'imagem' : 'video');
        $formato = $midia->formato ?? 'jpg';
        
        if ($request->hasFile('media_file')) {
            if ($path) { // Only delete if a path exists
                Storage::disk('public')->delete($path);
            }
            
            $extension = $request->file('media_file')->extension();
            $formato = in_array($extension, ['jpg', 'png', 'gif', 'mp4', 'html']) ? $extension : 'jpg';
            
            if ($request->type === 'image' || $tipo === 'imagem') {
                $manager = new ImageManager(new Driver());
                $image = $manager->read($request->file('media_file'));
                $image->resize(640, 1100);
                $filename = uniqid() . '.' . $extension;
                $path = 'midias/' . $filename;
                Storage::disk('public')->put($path, (string) $image->encode());
            } else {
                $path = $request->file('media_file')->store('midias', 'public');
            }
        }

        $midia->update([
            'title' => $request->title,
            'tipo' => $tipo,
            'formato' => $formato,
            'path' => $path,
            'empresa_legado_id' => $request->empresa_legado_id,
            'equipment_serial' => $request->equipment_serial,
            'position' => $request->position,
            'start_datetime' => $request->start_datetime,
            'end_datetime' => $request->end_datetime,
        ]);

        return redirect()->route('midias.index');
    }

    public function destroy(Midia $midia)
    {
        // Storage::disk('public')->delete($midia->path);
        $midia->delete();

        return redirect()->route('midias.index');
    }

    public function toggleStatus(Midia $midia)
    {
        $midia->status = !$midia->status;
        $midia->save();

        return redirect()->route('midias.index');
    }

    public function trash()
    {
        $midias = Midia::onlyTrashed()
                        ->with('destroyer:id,name')
                        ->latest('deleted_at')
                        ->get();
        return response()->json($midias);
    }

    public function restore($id)
    {
        $midia = Midia::onlyTrashed()->findOrFail($id);
        $midia->restore();

        return redirect()->route('midias.trash');
    }
}
