<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class AuditoriaSplash extends Model
{
    /**
     * Nome da tabela
     */
    protected $table = 'auditoria_splash';

    /**
     * Timestamps (apenas created_at)
     */
    const UPDATED_AT = null;

    /**
     * Campos fillable
     */
    protected $fillable = [
        'hs_serial',
        'cl_mac_address',
        'pag_click',
        'imagem',
        'dsc_identity',
        'created_at',
    ];

    /**
     * Casts
     */
    protected $casts = [
        'created_at' => 'datetime',
    ];

    /**
     * Scope: Filtrar por período
     */
    public function scopePorPeriodo($query, $dataInicio, $dataFim)
    {
        return $query->whereBetween('created_at', [$dataInicio, $dataFim]);
    }

    /**
     * Scope: Filtrar por hotspot serial
     */
    public function scopePorHotspot($query, $serial)
    {
        return $query->where('hs_serial', $serial);
    }

    /**
     * Scope: Filtrar por página clicada
     */
    public function scopePorPaginaClicada($query, $paginaClick)
    {
        return $query->where('pag_click', $paginaClick);
    }

    /**
     * Scope: Últimos N registros
     */
    public function scopeUltimos($query, $quantidade = 30)
    {
        return $query->latest('created_at')->limit($quantidade);
    }

    /**
     * Accessor: MAC address formatado (parcial para privacidade)
     */
    public function getMacFormatadoAttribute(): string
    {
        if (!$this->cl_mac_address) return '---';
        return substr($this->cl_mac_address, 0, 8) . '***' . substr($this->cl_mac_address, -2);
    }
}
