<?php

namespace App\Models;

use App\Traits\Userstamps;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class Criativo extends Model
{
    /** @use HasFactory<\Database\Factories\CriativoFactory> */
    use HasFactory, SoftDeletes, Userstamps;

    /**
     * Nome da tabela
     */
    protected $table = 'criativos';

    /**
     * Campos que podem ser preenchidos em massa
     */
    protected $fillable = [
        'franquia_id',
        'tipo',
        'title',
        'formato',
        'path',
        'conteudo_html',
        'status_aprovacao',
        'aprovado_por',
        'aprovado_em',
        'motivo_rejeicao',
        'bloqueado_permanente',
        'status',
        'position',
        'start_date',
        'end_date',
        'views_count',
    ];

    /**
     * Casting de atributos
     */
    protected $casts = [
        'start_date' => 'datetime',
        'end_date' => 'datetime',
        'aprovado_em' => 'datetime',
        'bloqueado_permanente' => 'boolean',
        'status' => 'boolean',
        'position' => 'integer',
        'views_count' => 'integer',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'deleted_at' => 'datetime',
    ];

    /**
     * Relacionamento: Criativo pertence a uma Franquia (Empresa)
     */
    public function franquia(): BelongsTo
    {
        return $this->belongsTo(Empresa::class, 'franquia_id');
    }

    /**
     * Relacionamento: Criativo foi aprovado por um User
     */
    public function aprovadoPor(): BelongsTo
    {
        return $this->belongsTo(User::class, 'aprovado_por');
    }

    /**
     * Relacionamento: Criativo tem muitas impressões
     */
    public function impressoes(): HasMany
    {
        return $this->hasMany(PublicacaoImpressao::class, 'criativo_id');
    }

    /**
     * Relacionamento: Criativo tem muitos cliques
     */
    public function cliques(): HasMany
    {
        return $this->hasMany(PublicacaoClique::class, 'criativo_id');
    }

    /**
     * Scopes para consultas comuns
     */
    public function scopeAtivos($query)
    {
        return $query->where('status', true);
    }

    public function scopeAprovados($query)
    {
        return $query->where('status_aprovacao', 'aprovado');
    }

    public function scopePendentes($query)
    {
        return $query->where('status_aprovacao', 'pendente');
    }

    public function scopePorTipo($query, string $tipo)
    {
        return $query->where('tipo', $tipo);
    }

    public function scopePorFormato($query, string $formato)
    {
        return $query->where('formato', $formato);
    }

    public function scopeDaFranquia($query, int $franquiaId)
    {
        return $query->where('franquia_id', $franquiaId);
    }

    public function scopeDoCatalogo($query)
    {
        return $query->whereNull('franquia_id');
    }
}
