<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class HotspotCustomization extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'hotspot_id',

        // Identidade Visual
        'brand_name',
        'logo_url',
        'favicon_url',

        // Cores - Background
        'bg_color_1',
        'bg_color_2',
        'bg_color_3',
        'bg_color_4',

        // Cores - Elementos
        'primary_color',
        'primary_hover_color',
        'text_color',
        'link_color',

        // Textos
        'welcome_title',
        'login_title',
        'register_title',
        'status_title',

        // Botões
        'btn_first_access',
        'btn_login',
        'btn_connect',
        'btn_disconnect',

        // Placeholders
        'placeholder_username',
        'placeholder_password',
        'placeholder_name',
        'placeholder_email',
        'placeholder_phone',

        // Tempos
        'modal_countdown_seconds',
        'banner_rotation_seconds',
        'status_refresh_seconds',

        // Propaganda
        'enable_advertisement',

        // URLs
        'terms_url',
        'privacy_url',
        'support_url',
        'redirect_after_login',

        // Campos do Formulário
        'require_name',
        'require_email',
        'require_phone',
        'require_birth_date',
        'require_address',

        // Tema
        'theme_preset',

        // Configurações Técnicas
        'enable_cpf_mask',
        'enable_phone_mask',
        'enable_password_hash',

        // Informações Exibidas
        'show_hotspot_identity',
        'show_mac_address',
        'show_ip_address',
        'show_upload_download',
        'show_session_time',

        // CSS/JS Customizado
        'custom_css',
        'custom_js',
        'custom_head_html',
        'custom_body_html',

        // Controle
        'is_active',
        'created_by',
        'updated_by',
        'deleted_by',
    ];

    protected $casts = [
        'enable_advertisement' => 'boolean',
        'require_name' => 'boolean',
        'require_email' => 'boolean',
        'require_phone' => 'boolean',
        'require_birth_date' => 'boolean',
        'require_address' => 'boolean',
        'enable_cpf_mask' => 'boolean',
        'enable_phone_mask' => 'boolean',
        'enable_password_hash' => 'boolean',
        'show_hotspot_identity' => 'boolean',
        'show_mac_address' => 'boolean',
        'show_ip_address' => 'boolean',
        'show_upload_download' => 'boolean',
        'show_session_time' => 'boolean',
        'is_active' => 'boolean',
        'modal_countdown_seconds' => 'integer',
        'banner_rotation_seconds' => 'integer',
        'status_refresh_seconds' => 'integer',
    ];

    /**
     * Relacionamento com Hotspot
     */
    public function hotspot(): BelongsTo
    {
        return $this->belongsTo(Hotspot::class);
    }

    /**
     * Usuário que criou o registro
     */
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Usuário que atualizou o registro
     */
    public function updater(): BelongsTo
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    /**
     * Usuário que deletou o registro
     */
    public function deleter(): BelongsTo
    {
        return $this->belongsTo(User::class, 'deleted_by');
    }

    /**
     * Retorna a customização como um objeto JavaScript para injetar no hotspot
     */
    public function toJavaScriptObject(): string
    {
        $data = $this->toArray();

        // Remover campos desnecessários
        unset($data['id'], $data['hotspot_id'], $data['created_at'], $data['updated_at'], $data['deleted_at']);
        unset($data['created_by'], $data['updated_by'], $data['deleted_by']);

        return json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    }

    /**
     * Gera o código JavaScript completo para carregar a customização
     */
    public function generateLoaderScript(): string
    {
        $customization = $this->toJavaScriptObject();

        return <<<JS
        <script>
        // 🎨 FreeFi Customization Loader (Auto-generated)
        (function() {
            window.freefiCustomization = $customization;

            // Aplicar customização automaticamente quando DOM carregar
            if (document.readyState === 'loading') {
                document.addEventListener('DOMContentLoaded', applyCustomization);
            } else {
                applyCustomization();
            }

            function applyCustomization() {
                const c = window.freefiCustomization;

                // Aplicar logo
                document.querySelectorAll('img[src*="logo.png"]').forEach(img => {
                    img.src = c.logo_url;
                });

                // Aplicar brand name
                document.querySelectorAll('h1').forEach(h1 => {
                    if (h1.textContent.includes('freefi.com.br')) {
                        h1.textContent = c.brand_name;
                    }
                    if (h1.textContent.includes('O Seu Wifi Grátis')) {
                        h1.textContent = c.welcome_title;
                    }
                    if (h1.textContent.includes('Cadastro de Usuário')) {
                        h1.textContent = c.register_title;
                    }
                    if (h1.textContent.includes('Você está conectado!')) {
                        h1.textContent = c.status_title;
                    }
                });

                // Aplicar cores via CSS injetado
                const style = document.createElement('style');
                style.innerHTML = `
                    body {
                        background: linear-gradient(135deg, \${c.bg_color_1} 0%, \${c.bg_color_2} 35%, \${c.bg_color_3} 65%, \${c.bg_color_4} 100%) !important;
                    }
                    input[type="submit"], .btn-primary {
                        background-color: \${c.primary_color} !important;
                    }
                    input[type="submit"]:hover, .btn-primary:hover {
                        background-color: \${c.primary_hover_color} !important;
                    }
                    h1, .info, table td {
                        color: \${c.text_color} !important;
                    }
                    a {
                        color: \${c.link_color} !important;
                    }
                `;
                document.head.appendChild(style);

                // Aplicar placeholders
                const placeholders = {
                    'cpf / login': c.placeholder_username,
                    'senha': c.placeholder_password,
                    'Nome': c.placeholder_name,
                    'Email': c.placeholder_email,
                    'Telefone': c.placeholder_phone
                };

                document.querySelectorAll('input[placeholder]').forEach(input => {
                    const oldPlaceholder = input.getAttribute('placeholder');
                    if (placeholders[oldPlaceholder]) {
                        input.setAttribute('placeholder', placeholders[oldPlaceholder]);
                    }
                });

                // Aplicar textos dos botões
                document.querySelectorAll('.btn-dark').forEach(btn => {
                    if (btn.textContent.includes('1º Acesso')) {
                        btn.textContent = c.btn_first_access;
                    }
                });
                document.querySelectorAll('.btn-primary').forEach(btn => {
                    if (btn.textContent.includes('Login')) {
                        btn.textContent = c.btn_login;
                    }
                });
                document.querySelectorAll('input[type="submit"]').forEach(btn => {
                    if (btn.value === 'Conectar') {
                        btn.value = c.btn_connect;
                    }
                    if (btn.value === 'Desconectar') {
                        btn.value = c.btn_disconnect;
                    }
                });

                // Aplicar tempo de countdown
                if (typeof window.countdownTime !== 'undefined') {
                    window.countdownTime = c.modal_countdown_seconds;
                }

                // Injetar CSS customizado (se existir)
                if (c.custom_css) {
                    const customStyle = document.createElement('style');
                    customStyle.innerHTML = c.custom_css;
                    document.head.appendChild(customStyle);
                }

                // Injetar HTML customizado no head (se existir)
                if (c.custom_head_html) {
                    document.head.insertAdjacentHTML('beforeend', c.custom_head_html);
                }

                // Injetar HTML customizado no body (se existir)
                if (c.custom_body_html) {
                    document.body.insertAdjacentHTML('afterbegin', c.custom_body_html);
                }

                console.log('✅ Customização FreeFi aplicada:', c.brand_name);
            }

            // Injetar JavaScript customizado (se existir)
            if (window.freefiCustomization.custom_js) {
                const script = document.createElement('script');
                script.textContent = window.freefiCustomization.custom_js;
                document.body.appendChild(script);
            }
        })();
        </script>
        JS;
    }

    /**
     * Scope para buscar customização ativa por serial do hotspot
     */
    public function scopeActiveBySerial($query, string $serial)
    {
        return $query->whereHas('hotspot', function ($q) use ($serial) {
            $q->where('serial', $serial);
        })->where('is_active', true)->latest();
    }
}
