<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class LogAcessoWifi extends Model
{
    /**
     * Nome da tabela
     */
    protected $table = 'logs_acesso_wifi';

    /**
     * Timestamps personalizados (apenas created_at)
     */
    const UPDATED_AT = null;

    /**
     * Campos fillable
     */
    protected $fillable = [
        'hotspot_id',
        'franquia_id',
        'cliente_id',
        'mac_address_hash',
        'ip_address',
        'user_agent',
        'publicacao_visualizada_id',
        'tempo_sessao',
        'dispositivo_tipo',
    ];

    /**
     * Casts
     */
    protected $casts = [
        'created_at' => 'datetime',
        'tempo_sessao' => 'integer',
    ];

    /**
     * Relacionamento com Hotspot
     */
    public function hotspot(): BelongsTo
    {
        return $this->belongsTo(Hotspot::class);
    }

    /**
     * Relacionamento com Franquia
     */
    public function franquia(): BelongsTo
    {
        return $this->belongsTo(Empresa::class, 'franquia_id');
    }

    /**
     * Relacionamento com Cliente
     */
    public function cliente(): BelongsTo
    {
        return $this->belongsTo(Empresa::class, 'cliente_id');
    }

    /**
     * Relacionamento com Publicação
     */
    public function publicacaoVisualizada(): BelongsTo
    {
        return $this->belongsTo(Publicacao::class, 'publicacao_visualizada_id');
    }

    /**
     * Scope: Logs de uma publicação específica
     */
    public function scopeDePublicacao($query, $publicacaoId)
    {
        return $query->where('publicacao_visualizada_id', $publicacaoId);
    }

    /**
     * Scope: Logs recentes
     */
    public function scopeRecentes($query, $dias = 30)
    {
        return $query->where('created_at', '>=', now()->subDays($dias));
    }

    /**
     * Accessor: Tipo de dispositivo formatado
     */
    public function getDispositivoFormatadoAttribute(): string
    {
        return match($this->dispositivo_tipo) {
            'mobile' => 'Celular',
            'desktop' => 'Desktop',
            'tablet' => 'Tablet',
            default => 'Outro',
        };
    }

    /**
     * Accessor: MAC parcial (últimos 4 dígitos do hash para privacidade)
     */
    public function getMacParcialAttribute(): string
    {
        return '***' . substr($this->mac_address_hash, -4);
    }
}
