<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class PublicacaoImpressao extends Model
{
    /** @use HasFactory<\Database\Factories\PublicacaoImpressaoFactory> */
    use HasFactory;

    /**
     * Nome da tabela
     */
    protected $table = 'publicacao_impressoes';

    /**
     * Campos que podem ser preenchidos em massa
     */
    protected $fillable = [
        'hotspot_id',
        'publicacao_id',
        'criativo_id',
        'data',
        'impressoes',
        'usuarios_unicos',
    ];

    /**
     * Casting de atributos
     */
    protected $casts = [
        'data' => 'date',
        'impressoes' => 'integer',
        'usuarios_unicos' => 'integer',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Valores padrão
     */
    protected $attributes = [
        'impressoes' => 0,
        'usuarios_unicos' => 0,
    ];

    /**
     * Relacionamento: Impressão pertence a um Hotspot
     */
    public function hotspot(): BelongsTo
    {
        return $this->belongsTo(Hotspot::class);
    }

    /**
     * Relacionamento: Impressão pertence a uma Publicação
     */
    public function publicacao(): BelongsTo
    {
        return $this->belongsTo(Publicacao::class);
    }

    /**
     * Relacionamento: Impressão pertence a um Criativo (antiga Mídia)
     */
    public function criativo(): BelongsTo
    {
        return $this->belongsTo(Criativo::class);
    }

    /**
     * Scopes para consultas comuns
     */
    public function scopePorData($query, $dataInicio, $dataFim = null)
    {
        if ($dataFim) {
            return $query->whereBetween('data', [$dataInicio, $dataFim]);
        }
        return $query->where('data', $dataInicio);
    }

    public function scopePorPublicacao($query, int $publicacaoId)
    {
        return $query->where('publicacao_id', $publicacaoId);
    }

    public function scopePorHotspot($query, int $hotspotId)
    {
        return $query->where('hotspot_id', $hotspotId);
    }

    /**
     * Calcula CTR (Click-Through Rate) - necessita join com cliques
     */
    public function getCtrAttribute(): ?float
    {
        if ($this->impressoes === 0) {
            return null;
        }

        $cliques = PublicacaoClique::where('hotspot_id', $this->hotspot_id)
            ->where('publicacao_id', $this->publicacao_id)
            ->where('criativo_id', $this->criativo_id)
            ->whereDate('clicked_at', $this->data)
            ->count();

        return ($cliques / $this->impressoes) * 100;
    }
}
