<?php

namespace App\Observers;

use App\Models\Empresa;
use Illuminate\Support\Str;

class EmpresaObserver
{
    /**
     * Handle the Empresa "creating" event.
     * Gera código de convite antes de salvar no banco
     */
    public function creating(Empresa $empresa): void
    {
        // Só gera se não tiver código
        if (empty($empresa->codigo_convite)) {
            $empresa->codigo_convite = $this->gerarCodigoUnico();
        }
    }

    /**
     * Gerar código de convite único
     * Formato: EGU2160261 (10 caracteres alfanuméricos)
     * Exemplo: ABC1234567, XYZ9876543
     */
    private function gerarCodigoUnico(): string
    {
        $tentativas = 0;
        $maxTentativas = 100;

        do {
            $codigo = $this->gerarCodigo();
            $tentativas++;
            
            if ($tentativas >= $maxTentativas) {
                throw new \RuntimeException('Não foi possível gerar código único');
            }
            
        } while (Empresa::where('codigo_convite', $codigo)->exists());

        return $codigo;
    }

    /**
     * Gerar código aleatório (10 caracteres alfanuméricos)
     * 3 letras maiúsculas + 7 números
     */
    private function gerarCodigo(): string
    {
        $letras = strtoupper(Str::random(3)); // ABC, XYZ, EGU
        $numeros = str_pad(rand(0, 9999999), 7, '0', STR_PAD_LEFT); // 2160261
        
        return $letras . $numeros; // EGU2160261
    }
}
