#!/usr/bin/env php
<?php

/**
 * Auditoria Completa do Sistema
 * ==============================
 * Verifica views, rotas, controllers e integridade geral
 */

require __DIR__.'/vendor/autoload.php';

$app = require_once __DIR__.'/bootstrap/app.php';
$kernel = $app->make('Illuminate\Contracts\Console\Kernel');
$kernel->bootstrap();

use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\View;

echo "\n";
echo "╔═══════════════════════════════════════════════════════════╗\n";
echo "║         AUDITORIA COMPLETA DO SISTEMA - FASE 2            ║\n";
echo "╚═══════════════════════════════════════════════════════════╝\n";
echo "\n";

$errors = [];
$warnings = [];
$success = [];

// ============================================================================
// 1. VERIFICAR VIEWS
// ============================================================================

echo "📄 VERIFICANDO VIEWS\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

$viewsToCheck = [
    // Auth
    'auth.login' => 'resources/views/auth/login.blade.php',
    'auth.register' => 'resources/views/auth/register.blade.php',
    'auth.forgot-password' => 'resources/views/auth/forgot-password.blade.php',
    
    // Dashboard
    'dashboard' => 'resources/views/dashboard.blade.php',
    
    // Users
    'users.index' => 'resources/views/users/index.blade.php',
    'users.create' => 'resources/views/users/create.blade.php',
    'users.edit' => 'resources/views/users/edit.blade.php',
    
    // Empresas
    'empresas.index' => 'resources/views/empresas/index.blade.php',
    'empresas.create' => 'resources/views/empresas/create.blade.php',
    'empresas.edit' => 'resources/views/empresas/edit.blade.php',
    
    // Mídias
    'midias.index' => 'resources/views/midias/index.blade.php',
    'midias.create' => 'resources/views/midias/create.blade.php',
    'midias.edit' => 'resources/views/midias/edit.blade.php',
    
    // Emails (Mailable)
    'emails.cadastro-aprovado' => 'resources/views/emails/cadastro-aprovado.blade.php',
    'emails.cadastro-rejeitado' => 'resources/views/emails/cadastro-rejeitado.blade.php',
    'emails.novo-cadastro-pendente' => 'resources/views/emails/novo-cadastro-pendente.blade.php',
    
    // Layouts
    'layouts.master' => 'resources/views/layouts/master.blade.php',
    'layouts.guest' => 'resources/views/layouts/guest.blade.php',
];

$viewsFound = 0;
$viewsMissing = 0;

foreach ($viewsToCheck as $viewName => $filePath) {
    $fullPath = base_path($filePath);
    
    if (File::exists($fullPath)) {
        $viewsFound++;
        $success[] = "✅ View: {$viewName}";
        echo "  ✅ {$viewName}\n";
        
        // Verificar se view tem conteúdo
        $content = File::get($fullPath);
        if (strlen($content) < 50) {
            $warnings[] = "⚠️  View {$viewName} parece estar vazia ou incompleta";
            echo "     ⚠️  Arquivo parece vazio\n";
        }
        
        // Verificar sintaxe básica Blade
        if (str_contains($viewName, 'layouts.')) {
            if (!str_contains($content, '@yield') && !str_contains($content, '{{ $slot }}')) {
                $warnings[] = "⚠️  Layout {$viewName} não tem @yield ou slot";
            }
        }
        
    } else {
        $viewsMissing++;
        $errors[] = "❌ View {$viewName} não encontrada em {$filePath}";
        echo "  ❌ {$viewName} - NÃO ENCONTRADA\n";
    }
}

echo "\n";
echo "  Total: {$viewsFound} encontradas, {$viewsMissing} faltando\n";
echo "\n";

// ============================================================================
// 2. VERIFICAR ROTAS
// ============================================================================

echo "🛣️  VERIFICANDO ROTAS\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

$routesToCheck = [
    // Auth (Breeze)
    ['method' => 'GET', 'name' => 'login', 'uri' => 'login'],
    ['method' => 'POST', 'name' => 'logout', 'uri' => 'logout'],
    ['method' => 'GET', 'name' => 'register', 'uri' => 'register'],
    ['method' => 'POST', 'name' => 'register.validate-code', 'uri' => 'register/validate-code'],
    
    // Dashboard
    ['method' => 'GET', 'name' => 'dashboard', 'uri' => 'dashboard'],
    
    // Users
    ['method' => 'GET', 'name' => 'users.index', 'uri' => 'users'],
    ['method' => 'GET', 'name' => 'users.create', 'uri' => 'users/create'],
    ['method' => 'POST', 'name' => 'users.store', 'uri' => 'users'],
    ['method' => 'GET', 'name' => 'users.edit', 'uri' => 'users/{user}/edit'],
    ['method' => 'PUT', 'name' => 'users.update', 'uri' => 'users/{user}'],
    ['method' => 'DELETE', 'name' => 'users.destroy', 'uri' => 'users/{user}'],
    ['method' => 'PATCH', 'name' => 'users.approve', 'uri' => 'users/{user}/approve'],
    ['method' => 'PATCH', 'name' => 'users.reject', 'uri' => 'users/{user}/reject'],
    
    // Empresas
    ['method' => 'GET', 'name' => 'empresas.index', 'uri' => 'empresas'],
    ['method' => 'GET', 'name' => 'empresas.create', 'uri' => 'empresas/create'],
    ['method' => 'POST', 'name' => 'empresas.store', 'uri' => 'empresas'],
    ['method' => 'GET', 'name' => 'empresas.edit', 'uri' => 'empresas/{empresa}/edit'],
    ['method' => 'PUT', 'name' => 'empresas.update', 'uri' => 'empresas/{empresa}'],
    ['method' => 'DELETE', 'name' => 'empresas.destroy', 'uri' => 'empresas/{empresa}'],
    
    // Mídias
    ['method' => 'GET', 'name' => 'midias.index', 'uri' => 'midias'],
    ['method' => 'GET', 'name' => 'midias.create', 'uri' => 'midias/create'],
    ['method' => 'POST', 'name' => 'midias.store', 'uri' => 'midias'],
    ['method' => 'GET', 'name' => 'midias.edit', 'uri' => 'midias/{midia}/edit'],
    ['method' => 'PUT', 'name' => 'midias.update', 'uri' => 'midias/{midia}'],
    ['method' => 'DELETE', 'name' => 'midias.destroy', 'uri' => 'midias/{midia}'],
];

$routesFound = 0;
$routesMissing = 0;

$allRoutes = collect(Route::getRoutes())->map(function ($route) {
    return [
        'method' => implode('|', $route->methods()),
        'uri' => $route->uri(),
        'name' => $route->getName(),
        'action' => $route->getActionName(),
    ];
});

foreach ($routesToCheck as $routeCheck) {
    $found = $allRoutes->first(function ($route) use ($routeCheck) {
        return $route['name'] === $routeCheck['name'] &&
               str_contains($route['method'], $routeCheck['method']);
    });
    
    if ($found) {
        $routesFound++;
        $success[] = "✅ Rota: {$routeCheck['method']} {$routeCheck['name']}";
        echo "  ✅ {$routeCheck['method']} {$routeCheck['name']} → {$found['uri']}\n";
    } else {
        $routesMissing++;
        $errors[] = "❌ Rota {$routeCheck['method']} {$routeCheck['name']} não encontrada";
        echo "  ❌ {$routeCheck['method']} {$routeCheck['name']} - NÃO ENCONTRADA\n";
    }
}

echo "\n";
echo "  Total: {$routesFound} encontradas, {$routesMissing} faltando\n";
echo "\n";

// ============================================================================
// 3. VERIFICAR CONTROLLERS
// ============================================================================

echo "🎮 VERIFICANDO CONTROLLERS\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

$controllersToCheck = [
    'App\\Http\\Controllers\\Auth\\AuthenticatedSessionController' => [
        'file' => 'app/Http/Controllers/Auth/AuthenticatedSessionController.php',
        'methods' => ['create', 'store', 'destroy'],
    ],
    'App\\Http\\Controllers\\Auth\\RegisterController' => [
        'file' => 'app/Http/Controllers/Auth/RegisterController.php',
        'methods' => ['showForm', 'register', 'validateCode'],
    ],
    'App\\Http\\Controllers\\UserController' => [
        'file' => 'app/Http/Controllers/UserController.php',
        'methods' => ['index', 'create', 'store', 'edit', 'update', 'destroy', 'approve', 'reject'],
    ],
    'App\\Http\\Controllers\\EmpresaController' => [
        'file' => 'app/Http/Controllers/EmpresaController.php',
        'methods' => ['index', 'create', 'store', 'edit', 'update', 'destroy'],
    ],
    'App\\Http\\Controllers\\MidiaController' => [
        'file' => 'app/Http/Controllers/MidiaController.php',
        'methods' => ['index', 'create', 'store', 'edit', 'update', 'destroy'],
    ],
];

$controllersFound = 0;
$controllersMissing = 0;

foreach ($controllersToCheck as $className => $info) {
    $fullPath = base_path($info['file']);
    
    if (File::exists($fullPath)) {
        $controllersFound++;
        echo "  ✅ {$className}\n";
        
        // Verificar se classe existe
        if (class_exists($className)) {
            $success[] = "✅ Controller: {$className}";
            
            // Verificar métodos
            $reflection = new ReflectionClass($className);
            $existingMethods = array_map(function ($method) {
                return $method->getName();
            }, $reflection->getMethods(ReflectionMethod::IS_PUBLIC));
            
            foreach ($info['methods'] as $methodName) {
                if (in_array($methodName, $existingMethods)) {
                    echo "     ✅ método {$methodName}()\n";
                } else {
                    $warnings[] = "⚠️  Método {$methodName}() não encontrado em {$className}";
                    echo "     ⚠️  método {$methodName}() - NÃO ENCONTRADO\n";
                }
            }
        } else {
            $errors[] = "❌ Classe {$className} não pode ser carregada";
            echo "     ❌ Classe não pode ser carregada\n";
        }
        
    } else {
        $controllersMissing++;
        $errors[] = "❌ Controller {$className} não encontrado em {$info['file']}";
        echo "  ❌ {$className} - ARQUIVO NÃO ENCONTRADO\n";
    }
}

echo "\n";
echo "  Total: {$controllersFound} encontrados, {$controllersMissing} faltando\n";
echo "\n";

// ============================================================================
// 4. VERIFICAR MAILABLES
// ============================================================================

echo "📧 VERIFICANDO MAILABLES (Emails)\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

$mailablesToCheck = [
    'App\\Mail\\CadastroAprovado' => [
        'file' => 'app/Mail/CadastroAprovado.php',
        'view' => 'emails.cadastro-aprovado',
    ],
    'App\\Mail\\CadastroRejeitado' => [
        'file' => 'app/Mail/CadastroRejeitado.php',
        'view' => 'emails.cadastro-rejeitado',
    ],
    'App\\Mail\\NovoCadastroPendente' => [
        'file' => 'app/Mail/NovoCadastroPendente.php',
        'view' => 'emails.novo-cadastro-pendente',
    ],
];

$mailablesFound = 0;
$mailablesMissing = 0;

foreach ($mailablesToCheck as $className => $info) {
    $fullPath = base_path($info['file']);
    
    if (File::exists($fullPath) && class_exists($className)) {
        $mailablesFound++;
        $success[] = "✅ Mailable: {$className}";
        echo "  ✅ {$className}\n";
        
        // Verificar se view existe
        $viewPath = str_replace('.', '/', $info['view']) . '.blade.php';
        $viewFullPath = resource_path('views/' . $viewPath);
        
        if (File::exists($viewFullPath)) {
            echo "     ✅ View: {$info['view']}\n";
        } else {
            $errors[] = "❌ View {$info['view']} não encontrada para {$className}";
            echo "     ❌ View: {$info['view']} - NÃO ENCONTRADA\n";
        }
        
        // Verificar se implementa ShouldQueue
        $content = File::get($fullPath);
        if (str_contains($content, 'implements ShouldQueue')) {
            echo "     ✅ Implementa ShouldQueue (envio em fila)\n";
        } else {
            $warnings[] = "⚠️  {$className} não implementa ShouldQueue (envio síncrono)";
            echo "     ⚠️  Não implementa ShouldQueue\n";
        }
        
    } else {
        $mailablesMissing++;
        $errors[] = "❌ Mailable {$className} não encontrado";
        echo "  ❌ {$className} - NÃO ENCONTRADO\n";
    }
}

echo "\n";
echo "  Total: {$mailablesFound} encontrados, {$mailablesMissing} faltando\n";
echo "\n";

// ============================================================================
// 5. VERIFICAR MODELS
// ============================================================================

echo "🗄️  VERIFICANDO MODELS\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

$modelsToCheck = [
    'App\\Models\\User' => [
        'file' => 'app/Models/User.php',
        'traits' => ['HasFactory', 'Notifiable'],
        'columns' => ['name', 'email', 'password', 'empresa_id', 'status_cadastro'],
    ],
    'App\\Models\\Empresa' => [
        'file' => 'app/Models/Empresa.php',
        'traits' => ['HasFactory', 'SoftDeletes', 'Userstamps'],
        'columns' => ['name', 'codigo_convite', 'tipo_empresa', 'status'],
    ],
    'App\\Models\\Midia' => [
        'file' => 'app/Models/Midia.php',
        'traits' => ['HasFactory', 'SoftDeletes', 'Userstamps'],
        'columns' => ['titulo', 'tipo', 'path', 'status'],
    ],
];

$modelsFound = 0;
$modelsMissing = 0;

foreach ($modelsToCheck as $className => $info) {
    $fullPath = base_path($info['file']);
    
    if (File::exists($fullPath) && class_exists($className)) {
        $modelsFound++;
        $success[] = "✅ Model: {$className}";
        echo "  ✅ {$className}\n";
        
        // Verificar traits
        $reflection = new ReflectionClass($className);
        $traits = array_keys($reflection->getTraits());
        
        foreach ($info['traits'] as $trait) {
            $traitFound = false;
            foreach ($traits as $usedTrait) {
                if (str_contains($usedTrait, $trait)) {
                    $traitFound = true;
                    break;
                }
            }
            
            if ($traitFound) {
                echo "     ✅ Trait: {$trait}\n";
            } else {
                $warnings[] = "⚠️  Trait {$trait} não encontrada em {$className}";
                echo "     ⚠️  Trait: {$trait} - NÃO ENCONTRADA\n";
            }
        }
        
    } else {
        $modelsMissing++;
        $errors[] = "❌ Model {$className} não encontrado";
        echo "  ❌ {$className} - NÃO ENCONTRADO\n";
    }
}

echo "\n";
echo "  Total: {$modelsFound} encontrados, {$modelsMissing} faltando\n";
echo "\n";

// ============================================================================
// 6. VERIFICAR MIDDLEWARE
// ============================================================================

echo "🛡️  VERIFICANDO MIDDLEWARE\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

$middlewareToCheck = [
    'App\\Http\\Middleware\\CheckApprovalStatus' => [
        'file' => 'app/Http/Middleware/CheckApprovalStatus.php',
        'alias' => 'check.approval',
    ],
];

$middlewareFound = 0;
$middlewareMissing = 0;

foreach ($middlewareToCheck as $className => $info) {
    $fullPath = base_path($info['file']);
    
    if (File::exists($fullPath) && class_exists($className)) {
        $middlewareFound++;
        $success[] = "✅ Middleware: {$className}";
        echo "  ✅ {$className}\n";
        echo "     Alias: {$info['alias']}\n";
    } else {
        $middlewareMissing++;
        $errors[] = "❌ Middleware {$className} não encontrado";
        echo "  ❌ {$className} - NÃO ENCONTRADO\n";
    }
}

echo "\n";
echo "  Total: {$middlewareFound} encontrados, {$middlewareMissing} faltando\n";
echo "\n";

// ============================================================================
// 7. VERIFICAR CONFIGURAÇÕES
// ============================================================================

echo "⚙️  VERIFICANDO CONFIGURAÇÕES\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

$configsToCheck = [
    'config/features.php' => 'Feature Flags',
    'config/mail.php' => 'Configuração de Email',
    'config/queue.php' => 'Configuração de Fila',
];

$configsFound = 0;
$configsMissing = 0;

foreach ($configsToCheck as $file => $description) {
    $fullPath = base_path($file);
    
    if (File::exists($fullPath)) {
        $configsFound++;
        $success[] = "✅ Config: {$description}";
        echo "  ✅ {$description} ({$file})\n";
    } else {
        $configsMissing++;
        $errors[] = "❌ Config {$description} não encontrado ({$file})";
        echo "  ❌ {$description} - NÃO ENCONTRADO\n";
    }
}

echo "\n";
echo "  Total: {$configsFound} encontrados, {$configsMissing} faltando\n";
echo "\n";

// ============================================================================
// RESUMO FINAL
// ============================================================================

echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "📊 RESUMO DA AUDITORIA\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

$totalSuccess = count($success);
$totalWarnings = count($warnings);
$totalErrors = count($errors);
$totalItems = $totalSuccess + $totalWarnings + $totalErrors;

echo "  ✅ Sucessos: {$totalSuccess}\n";
echo "  ⚠️  Avisos: {$totalWarnings}\n";
echo "  ❌ Erros: {$totalErrors}\n";
echo "\n";

if ($totalErrors > 0) {
    echo "❌ ERROS CRÍTICOS ENCONTRADOS:\n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    foreach ($errors as $error) {
        echo "  {$error}\n";
    }
    echo "\n";
}

if ($totalWarnings > 0) {
    echo "⚠️  AVISOS (não críticos):\n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    foreach ($warnings as $warning) {
        echo "  {$warning}\n";
    }
    echo "\n";
}

// Status geral
$percentage = $totalItems > 0 ? round(($totalSuccess / $totalItems) * 100, 1) : 0;

echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "🎯 STATUS GERAL: {$percentage}%\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

if ($totalErrors === 0) {
    echo "  🎉 SISTEMA PRONTO PARA PRODUÇÃO!\n\n";
    exit(0);
} else {
    echo "  ⚠️  CORRIJA OS ERROS ANTES DE CONTINUAR\n\n";
    exit(1);
}
