<?php

namespace Tests\Browser;

use Laravel\Dusk\Browser;
use Tests\DuskTestCase;

class ProducaoPaginacaoTest extends DuskTestCase
{
    /**
     * Teste: Análise completa da paginação em produção
     */
    public function testAnalisarPaginacaoProducao()
    {
        $this->browse(function (Browser $browser) {
            // Acessar servidor de produção
            $browser->visit('http://172.233.21.57:8082/login')
                    ->screenshot('prod-01-login-page');

            dump('=== INICIANDO ANÁLISE DE PRODUÇÃO ===');

            // Fazer login
            try {
                $browser->type('email', 'linuxewireless@gmail.com')
                        ->type('password', 'password')
                        ->screenshot('prod-02-antes-login')
                        ->press('Entrar')
                        ->waitForLocation('/dashboard', 15)
                        ->screenshot('prod-03-dashboard');

                dump('✅ Login realizado com sucesso');
            } catch (\Exception $e) {
                dump('❌ Erro no login:', $e->getMessage());
                $browser->screenshot('prod-erro-login');
                return;
            }

            // Navegar para publicações
            try {
                $browser->visit('http://172.233.21.57:8082/publicacoes')
                        ->waitFor('table', 15)
                        ->screenshot('prod-04-publicacoes-page');

                dump('✅ Página de publicações carregada');
            } catch (\Exception $e) {
                dump('❌ Erro ao carregar publicações:', $e->getMessage());
                $browser->screenshot('prod-erro-publicacoes');
                return;
            }

            // ANÁLISE 1: Verificar se paginação existe
            $paginacaoExiste = $browser->script("return !!document.querySelector('.pagination');");
            dump('📊 Paginação existe no DOM:', $paginacaoExiste ? 'SIM' : 'NÃO');

            if ($paginacaoExiste) {
                $browser->screenshot('prod-05-paginacao-encontrada');

                // HTML da paginação
                $paginacaoHtml = $browser->script("return document.querySelector('.pagination')?.outerHTML;");
                dump('📄 HTML da Paginação:', substr($paginacaoHtml[0] ?? '', 0, 500) . '...');

                // Classes Bootstrap
                $classes = $browser->script("
                    const pag = document.querySelector('.pagination');
                    if (pag) {
                        return {
                            classes: pag.className,
                            hasPageItem: pag.querySelectorAll('.page-item').length,
                            hasPageLink: pag.querySelectorAll('.page-link').length,
                            hasActive: pag.querySelectorAll('.page-item.active').length,
                            structure: pag.tagName
                        };
                    }
                    return null;
                ");
                dump('🎨 Classes da paginação:', $classes);

                // Verificar se é Bootstrap 5
                $isBootstrap5 = $browser->script("
                    const pag = document.querySelector('.pagination');
                    if (!pag) return false;

                    const hasPageItem = pag.querySelector('.page-item') !== null;
                    const hasPageLink = pag.querySelector('.page-link') !== null;
                    const isInNav = pag.closest('nav') !== null;

                    return hasPageItem && hasPageLink && isInNav;
                ");
                dump('✅ É Bootstrap 5:', $isBootstrap5 ? 'SIM' : 'NÃO');

                // Estilos CSS aplicados
                $estilos = $browser->script("
                    const pag = document.querySelector('.pagination');
                    if (pag) {
                        const computed = window.getComputedStyle(pag);
                        return {
                            display: computed.display,
                            visibility: computed.visibility,
                            opacity: computed.opacity,
                            color: computed.color,
                            backgroundColor: computed.backgroundColor
                        };
                    }
                    return null;
                ");
                dump('🎨 Estilos CSS:', $estilos);

                // Verificar problemas visuais
                $problemas = $browser->script("
                    const pag = document.querySelector('.pagination');
                    const issues = [];

                    if (pag) {
                        const computed = window.getComputedStyle(pag);

                        if (computed.display === 'none') issues.push('❌ display: none');
                        if (computed.visibility === 'hidden') issues.push('❌ visibility: hidden');
                        if (computed.opacity === '0') issues.push('❌ opacity: 0');
                        if (computed.color === computed.backgroundColor) {
                            issues.push('❌ Texto invisível (cor = fundo)');
                        }

                        if (issues.length === 0) {
                            issues.push('✅ Sem problemas visuais detectados');
                        }
                    }

                    return issues;
                ");
                dump('🔍 Problemas detectados:', $problemas);

                // Informações de paginação (Showing X to Y of Z)
                $info = $browser->script("
                    const nav = document.querySelector('nav');
                    if (nav) {
                        const p = nav.querySelector('p');
                        return p ? p.textContent.trim() : 'Não encontrado';
                    }
                    return 'Nav não encontrado';
                ");
                dump('ℹ️ Informações de paginação:', $info);

                // Testar clique no próximo (se existir)
                $temProximo = $browser->script("
                    return !!document.querySelector('.page-link[rel=\"next\"]');
                ");

                if ($temProximo) {
                    dump('🔗 Botão "Próximo" encontrado, tentando clicar...');
                    $browser->screenshot('prod-06-antes-proximo');

                    try {
                        $browser->click('.page-link[rel="next"]')
                                ->pause(2000)
                                ->screenshot('prod-07-depois-proximo');

                        $urlAtual = $browser->script("return window.location.href;");
                        dump('✅ Navegação funcionou! URL:', $urlAtual);
                    } catch (\Exception $e) {
                        dump('❌ Erro ao clicar no próximo:', $e->getMessage());
                        $browser->screenshot('prod-erro-proximo');
                    }
                } else {
                    dump('ℹ️ Botão "Próximo" não disponível (última página ou poucas páginas)');
                }

            } else {
                $browser->screenshot('prod-05-paginacao-nao-encontrada');
                dump('⚠️ PAGINAÇÃO NÃO ENCONTRADA NO DOM');
            }

            // ANÁLISE 2: Contar registros
            $totalRegistros = $browser->script("
                return document.querySelectorAll('table tbody tr').length;
            ");
            dump('📊 Total de linhas na tabela:', $totalRegistros);

            // ANÁLISE 3: Verificar se condição hasPages() está OK
            $hasPages = $browser->script("
                // Tentar detectar se há múltiplas páginas
                const table = document.querySelector('table');
                const rows = table ? table.querySelectorAll('tbody tr').length : 0;
                const pagination = document.querySelector('.pagination');

                return {
                    rows: rows,
                    hasPagination: !!pagination,
                    shouldHavePagination: rows >= 15
                };
            ");
            dump('📋 Análise hasPages():', $hasPages);

            // ANÁLISE 4: Verificar se AppServiceProvider está configurado
            $bootstrapConfig = $browser->script("
                // Verificar se Bootstrap está sendo usado
                const pag = document.querySelector('.pagination');
                if (!pag) return { configured: false, reason: 'Paginação não encontrada' };

                // Bootstrap 5 usa .page-item e .page-link
                const hasBS5Classes = pag.querySelector('.page-item') !== null;

                // Tailwind usa classes diferentes
                const hasTailwindClasses = pag.className.includes('flex') ||
                                           pag.className.includes('space-x');

                return {
                    configured: hasBS5Classes,
                    framework: hasBS5Classes ? 'Bootstrap 5' : (hasTailwindClasses ? 'Tailwind' : 'Desconhecido'),
                    classes: pag.className
                };
            ");
            dump('⚙️ Configuração do framework:', $bootstrapConfig);

            // ANÁLISE 5: Verificar arquivos CSS carregados
            $cssFiles = $browser->script("
                return Array.from(document.querySelectorAll('link[rel=\"stylesheet\"]'))
                    .map(link => ({
                        href: link.href,
                        hasBootstrap: link.href.includes('bootstrap')
                    }))
                    .filter(file => file.hasBootstrap || file.href.includes('app'));
            ");
            dump('📁 Arquivos CSS relevantes:', $cssFiles);

            // Screenshot final
            $browser->screenshot('prod-08-analise-final');

            dump('=== ANÁLISE DE PRODUÇÃO CONCLUÍDA ===');
        });
    }

    /**
     * Teste: Verificar se correção foi aplicada em produção
     */
    public function testVerificarCorrecaoAplicada()
    {
        $this->browse(function (Browser $browser) {
            dump('=== VERIFICANDO SE CORREÇÃO FOI APLICADA ===');

            // Login
            $browser->visit('http://172.233.21.57:8082/login')
                    ->type('email', 'linuxewireless@gmail.com')
                    ->type('password', 'password')
                    ->press('Entrar')
                    ->waitForLocation('/dashboard', 15);

            // Ir para publicações
            $browser->visit('http://172.233.21.57:8082/publicacoes')
                    ->waitFor('table', 15);

            // Verificar se tem paginação
            $temPaginacao = $browser->script("return !!document.querySelector('.pagination');");

            if (!$temPaginacao) {
                dump('⚠️ Paginação não encontrada - possivelmente não há dados suficientes (< 16 registros)');
                dump('ℹ️ Isso é NORMAL se houver menos de 16 publicações');

                $total = $browser->script("return document.querySelectorAll('table tbody tr').length;");
                dump('📊 Total de registros visíveis:', $total);

                if ($total[0] < 15) {
                    dump('✅ COMPORTAMENTO CORRETO: Paginação oculta quando há poucos registros');
                }
            } else {
                dump('✅ Paginação encontrada!');

                // Verificar se é Bootstrap 5
                $isBS5 = $browser->script("
                    const pag = document.querySelector('.pagination');
                    return pag &&
                           pag.querySelector('.page-item') !== null &&
                           pag.querySelector('.page-link') !== null;
                ");

                if ($isBS5) {
                    dump('✅ CORREÇÃO APLICADA: Bootstrap 5 configurado corretamente!');
                    dump('🎉 AppServiceProvider::boot() com Paginator::useBootstrapFive() está ativo');
                } else {
                    dump('❌ CORREÇÃO NÃO APLICADA: Ainda usando template padrão (Tailwind)');
                    dump('⚠️ É necessário adicionar Paginator::useBootstrapFive() no AppServiceProvider');
                }

                $browser->screenshot('prod-verificacao-correcao');
            }

            dump('=== VERIFICAÇÃO CONCLUÍDA ===');
        });
    }
}
