<?php

namespace Tests\Browser\Publicacao;

use Laravel\Dusk\Browser;
use Tests\DuskTestCase;
use App\Models\{User, Publicacao, Criativo, Empresa};
use Illuminate\Foundation\Testing\DatabaseMigrations;

/**
 * Testes Dusk Completos - Interface de Publicações
 * 
 * Testa a interface real no navegador, incluindo JavaScript,
 * formulários dinâmicos, modais e interações do usuário
 */
class PublicacaoDuskCompleteTest extends DuskTestCase
{
    use DatabaseMigrations;

    protected $user;
    protected $franquia;
    protected $cliente;
    protected $criativo;

    protected function setUp(): void
    {
        parent::setUp();

        $this->user = User::factory()->create();
        $this->franquia = Empresa::factory()->create(['tipo_empresa' => 'franquia']);
        $this->cliente = Empresa::factory()->create(['tipo_empresa' => 'cliente']);
        $this->criativo = Criativo::factory()->create([
            'status_aprovacao' => 'aprovado',
            'status' => 0,
            'title' => 'Criativo Teste Dusk'
        ]);
    }

    /**
     * Teste: Navegar para página de publicações
     */
    public function test_pode_acessar_pagina_de_publicacoes()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->user)
                    ->visit('/publicacoes')
                    ->assertSee('Publicações')
                    ->assertSee('Nova Publicação')
                    ->screenshot('publicacoes-index');
        });
    }

    /**
     * Teste: Cards de estatísticas estão visíveis
     */
    public function test_exibe_cards_de_estatisticas()
    {
        Publicacao::factory()->count(5)->create([
            'status' => 'ativa',
            'franquia_id' => $this->franquia->id,
            'cliente_id' => $this->cliente->id,
        ]);

        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->user)
                    ->visit('/publicacoes')
                    ->assertSeeIn('.stats-card', 'Ativas')
                    ->assertSeeIn('.stats-card', 'Pausadas')
                    ->assertSeeIn('.stats-card', 'Expiradas')
                    ->screenshot('publicacoes-stats');
        });
    }

    /**
     * Teste: Paginação funciona corretamente
     */
    public function test_paginacao_funciona()
    {
        Publicacao::factory()->count(20)->create([
            'franquia_id' => $this->franquia->id,
            'cliente_id' => $this->cliente->id,
        ]);

        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->user)
                    ->visit('/publicacoes')
                    ->assertPresent('.pagination')
                    ->click('.pagination a[rel="next"]')
                    ->waitForLocation('/publicacoes?page=2')
                    ->assertQueryStringHas('page', '2')
                    ->screenshot('publicacoes-paginacao');
        });
    }

    /**
     * Teste: Filtros funcionam
     */
    public function test_filtros_funcionam()
    {
        Publicacao::factory()->create([
            'status' => 'ativa',
            'titulo' => 'Publicação Ativa',
            'franquia_id' => $this->franquia->id,
            'cliente_id' => $this->cliente->id,
        ]);

        Publicacao::factory()->create([
            'status' => 'pausada',
            'titulo' => 'Publicação Pausada',
            'franquia_id' => $this->franquia->id,
            'cliente_id' => $this->cliente->id,
        ]);

        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->user)
                    ->visit('/publicacoes')
                    ->select('status', 'ativa')
                    ->press('Filtrar')
                    ->waitForText('Publicação Ativa')
                    ->assertSee('Publicação Ativa')
                    ->assertDontSee('Publicação Pausada')
                    ->screenshot('publicacoes-filtro-status');
        });
    }

    /**
     * Teste: Formulário de criação carrega corretamente
     */
    public function test_formulario_criacao_carrega()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->user)
                    ->visit('/publicacoes/create')
                    ->assertSee('Nova Publicação')
                    ->assertPresent('select[name="criativo_id"]')
                    ->assertPresent('select[name="franquia_id"]')
                    ->assertPresent('select[name="cliente_id"]')
                    ->assertPresent('input[name="titulo"]')
                    ->assertPresent('input[name="data_inicio"]')
                    ->assertPresent('input[name="data_fim"]')
                    ->screenshot('publicacoes-form-create');
        });
    }

    /**
     * Teste: Pode criar publicação através do formulário
     */
    public function test_pode_criar_publicacao_pelo_formulario()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->user)
                    ->visit('/publicacoes/create')
                    ->select('criativo_id', $this->criativo->id)
                    ->select('franquia_id', $this->franquia->id)
                    ->select('cliente_id', $this->cliente->id)
                    ->type('titulo', 'Nova Publicação Dusk')
                    ->type('data_inicio', now()->format('Y-m-d'))
                    ->type('data_fim', now()->addDays(30)->format('Y-m-d'))
                    ->type('link_destino', 'https://example.com')
                    ->check('habilitar_clique')
                    ->check('track_impressoes')
                    ->check('track_cliques')
                    ->type('valor_contrato', '1500.00')
                    ->type('impressoes_contratadas', '10000')
                    ->select('posicao', '1')
                    ->select('status', 'ativa')
                    ->select('tipo_alcance', 'todos')
                    ->press('Salvar')
                    ->waitForLocation('/publicacoes')
                    ->assertSee('Publicação criada com sucesso')
                    ->assertSee('Nova Publicação Dusk')
                    ->screenshot('publicacoes-criada');
        });

        $this->assertDatabaseHas('publicacoes', [
            'titulo' => 'Nova Publicação Dusk',
        ]);
    }

    /**
     * Teste: Validações do formulário funcionam
     */
    public function test_validacoes_formulario_funcionam()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->user)
                    ->visit('/publicacoes/create')
                    ->press('Salvar')
                    ->waitForText('campo')
                    ->assertSee('obrigatório')
                    ->screenshot('publicacoes-validacao');
        });
    }

    /**
     * Teste: Preview do criativo funciona
     */
    public function test_preview_criativo_funciona()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->user)
                    ->visit('/publicacoes/create')
                    ->select('criativo_id', $this->criativo->id)
                    ->pause(1000) // Aguardar carregamento AJAX
                    ->assertPresent('#preview-criativo')
                    ->screenshot('publicacoes-preview');
        });
    }

    /**
     * Teste: Pode adicionar walled garden dinamicamente
     */
    public function test_pode_adicionar_walled_garden()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->user)
                    ->visit('/publicacoes/create')
                    ->click('#btn-add-walled-garden')
                    ->pause(500)
                    ->assertPresent('.walled-garden-item')
                    ->select('.walled-garden-item select[name*="tipo"]', 'domain')
                    ->type('.walled-garden-item input[name*="endereco"]', 'example.com')
                    ->type('.walled-garden-item input[name*="descricao"]', 'Domínio teste')
                    ->screenshot('publicacoes-walled-garden');
        });
    }

    /**
     * Teste: Pode remover walled garden
     */
    public function test_pode_remover_walled_garden()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->user)
                    ->visit('/publicacoes/create')
                    ->click('#btn-add-walled-garden')
                    ->pause(500)
                    ->assertPresent('.walled-garden-item')
                    ->click('.walled-garden-item .btn-remove')
                    ->pause(500)
                    ->assertMissing('.walled-garden-item')
                    ->screenshot('publicacoes-walled-garden-removed');
        });
    }

    /**
     * Teste: Pode editar publicação existente
     */
    public function test_pode_editar_publicacao()
    {
        $publicacao = Publicacao::factory()->create([
            'franquia_id' => $this->franquia->id,
            'cliente_id' => $this->cliente->id,
            'titulo' => 'Título Original',
        ]);

        $this->browse(function (Browser $browser) use ($publicacao) {
            $browser->loginAs($this->user)
                    ->visit("/publicacoes/{$publicacao->id}/edit")
                    ->assertInputValue('titulo', 'Título Original')
                    ->clear('titulo')
                    ->type('titulo', 'Título Editado')
                    ->press('Atualizar')
                    ->waitForLocation('/publicacoes')
                    ->assertSee('Publicação atualizada com sucesso')
                    ->assertSee('Título Editado')
                    ->screenshot('publicacoes-editada');
        });
    }

    /**
     * Teste: Modal de analytics abre corretamente
     */
    public function test_modal_analytics_abre()
    {
        $publicacao = Publicacao::factory()->create([
            'franquia_id' => $this->franquia->id,
            'cliente_id' => $this->cliente->id,
        ]);

        $this->browse(function (Browser $browser) use ($publicacao) {
            $browser->loginAs($this->user)
                    ->visit('/publicacoes')
                    ->click('.btn-analytics')
                    ->waitFor('#modal-analytics')
                    ->assertVisible('#modal-analytics')
                    ->assertSeeIn('#modal-analytics', 'Estatísticas')
                    ->screenshot('publicacoes-modal-analytics');
        });
    }

    /**
     * Teste: Pode alternar status da publicação
     */
    public function test_pode_alternar_status()
    {
        $publicacao = Publicacao::factory()->create([
            'status' => 'ativa',
            'franquia_id' => $this->franquia->id,
            'cliente_id' => $this->cliente->id,
        ]);

        $this->browse(function (Browser $browser) use ($publicacao) {
            $browser->loginAs($this->user)
                    ->visit('/publicacoes')
                    ->click('.btn-toggle-status')
                    ->waitForText('pausada')
                    ->assertSee('pausada')
                    ->screenshot('publicacoes-status-alterado');
        });
    }

    /**
     * Teste: Pode deletar publicação com confirmação
     */
    public function test_pode_deletar_publicacao_com_confirmacao()
    {
        $publicacao = Publicacao::factory()->create([
            'franquia_id' => $this->franquia->id,
            'cliente_id' => $this->cliente->id,
            'titulo' => 'Para Deletar',
        ]);

        $this->browse(function (Browser $browser) use ($publicacao) {
            $browser->loginAs($this->user)
                    ->visit('/publicacoes')
                    ->click('.btn-delete')
                    ->waitForDialog()
                    ->acceptDialog()
                    ->waitForText('excluída com sucesso')
                    ->assertDontSee('Para Deletar')
                    ->screenshot('publicacoes-deletada');
        });
    }

    /**
     * Teste: Busca por título funciona
     */
    public function test_busca_por_titulo_funciona()
    {
        Publicacao::factory()->create([
            'titulo' => 'Publicação Específica',
            'franquia_id' => $this->franquia->id,
            'cliente_id' => $this->cliente->id,
        ]);

        Publicacao::factory()->create([
            'titulo' => 'Outra Publicação',
            'franquia_id' => $this->franquia->id,
            'cliente_id' => $this->cliente->id,
        ]);

        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->user)
                    ->visit('/publicacoes')
                    ->type('q', 'Específica')
                    ->press('Buscar')
                    ->waitForText('Publicação Específica')
                    ->assertSee('Publicação Específica')
                    ->assertDontSee('Outra Publicação')
                    ->screenshot('publicacoes-busca');
        });
    }

    /**
     * Teste: Página de estatísticas carrega gráficos
     */
    public function test_pagina_estatisticas_carrega_graficos()
    {
        $publicacao = Publicacao::factory()->create([
            'franquia_id' => $this->franquia->id,
            'cliente_id' => $this->cliente->id,
        ]);

        $this->browse(function (Browser $browser) use ($publicacao) {
            $browser->loginAs($this->user)
                    ->visit("/publicacoes/{$publicacao->id}/estatisticas")
                    ->assertSee('Estatísticas Detalhadas')
                    ->assertPresent('#grafico-impressoes')
                    ->assertPresent('#grafico-cliques-hora')
                    ->assertPresent('#tabela-top-hotspots')
                    ->screenshot('publicacoes-estatisticas-completas');
        });
    }

    /**
     * Teste: Responsividade mobile
     */
    public function test_responsividade_mobile()
    {
        $this->browse(function (Browser $browser) {
            $browser->resize(375, 667) // iPhone SE
                    ->loginAs($this->user)
                    ->visit('/publicacoes')
                    ->assertSee('Publicações')
                    ->screenshot('publicacoes-mobile');
        });
    }
}
