<?php

namespace Tests\Browser;

use App\Models\User;
use App\Models\Publicacao;
use Laravel\Dusk\Browser;
use Tests\DuskTestCase;

class PublicacaoAnalyticsModalFuncionalTest extends DuskTestCase
{
    /**
     * Testa se o modal de analytics carrega os dados corretamente
     */
    public function testModalCarregaDadosViaAjax(): void
    {
        $user = User::where('email', 'linuxewireless@gmail.com')->first();
        $publicacao = Publicacao::first();

        if (!$user || !$publicacao) {
            $this->markTestSkipped('Usuário ou publicação não encontrados no banco');
        }

        $this->browse(function (Browser $browser) use ($user, $publicacao) {
            $browser->loginAs($user)
                    ->visit('/publicacoes')
                    ->waitFor('table tbody tr', 10)
                    ->screenshot('publicacoes_antes_modal')

                    // Clicar no ícone de analytics
                    ->click('[data-publicacao-id="' . $publicacao->id . '"]')
                    ->waitFor('#analyticsModal.show', 5)
                    ->screenshot('modal_aberto')

                    // Verificar que o modal abriu
                    ->assertVisible('#analyticsModal')

                    // Esperar o loading desaparecer e os dados carregarem
                    ->waitUntilMissing('.spinner-border', 10)
                    ->screenshot('modal_dados_carregados')

                    // Verificar elementos de estatísticas
                    ->assertSeeIn('#analyticsModalBody', 'Impressões')
                    ->assertSeeIn('#analyticsModalBody', 'Cliques')
                    ->assertSeeIn('#analyticsModalBody', 'Média Diária')
                    ->assertSeeIn('#analyticsModalBody', 'Período')

                    // Verificar cards foram renderizados
                    ->assertPresent('#analyticsModalBody .card')

                    // Verificar link de estatísticas completas
                    ->assertAttribute('#verEstatisticasCompletas', 'href', '/publicacoes/' . $publicacao->id . '/estatisticas')

                    ->screenshot('modal_validado');
        });
    }

    /**
     * Testa se o modal exibe mensagem quando não há dados
     */
    public function testModalExibeMensagemSemDados(): void
    {
        $user = User::where('email', 'linuxewireless@gmail.com')->first();
        $publicacao = Publicacao::first();

        if (!$user || !$publicacao) {
            $this->markTestSkipped('Usuário ou publicação não encontrados no banco');
        }

        $this->browse(function (Browser $browser) use ($user, $publicacao) {
            $browser->loginAs($user)
                    ->visit('/publicacoes')
                    ->waitFor('table tbody tr', 10)
                    ->click('[data-publicacao-id="' . $publicacao->id . '"]')
                    ->waitFor('#analyticsModal.show', 5)
                    ->waitUntilMissing('.spinner-border', 10)

                    // Se não houver impressões, deve mostrar mensagem
                    ->pause(500)
                    ->screenshot('modal_verificacao_mensagem');

            // Verificar se mostra 0 impressões ou mensagem de sem dados
            $browser->assertSee('0');
        });
    }

    /**
     * Testa título do modal é atualizado corretamente
     */
    public function testModalAtualizaTitulo(): void
    {
        $user = User::where('email', 'linuxewireless@gmail.com')->first();
        $publicacao = Publicacao::first();

        if (!$user || !$publicacao) {
            $this->markTestSkipped('Usuário ou publicação não encontrados no banco');
        }

        $this->browse(function (Browser $browser) use ($user, $publicacao) {
            $browser->loginAs($user)
                    ->visit('/publicacoes')
                    ->waitFor('table tbody tr', 10)
                    ->click('[data-publicacao-id="' . $publicacao->id . '"]')
                    ->waitFor('#analyticsModal.show', 5)

                    // Verificar título contém nome da publicação
                    ->assertSeeIn('#analyticsModalLabel', 'Visualizações: ' . $publicacao->titulo);
        });
    }

    /**
     * Testa se o modal pode ser fechado normalmente
     */
    public function testModalFechaCorretamente(): void
    {
        $user = User::where('email', 'linuxewireless@gmail.com')->first();
        $publicacao = Publicacao::first();

        if (!$user || !$publicacao) {
            $this->markTestSkipped('Usuário ou publicação não encontrados no banco');
        }

        $this->browse(function (Browser $browser) use ($user, $publicacao) {
            $browser->loginAs($user)
                    ->visit('/publicacoes')
                    ->waitFor('table tbody tr', 10)
                    ->click('[data-publicacao-id="' . $publicacao->id . '"]')
                    ->waitFor('#analyticsModal.show', 5)
                    ->pause(500)

                    // Fechar modal
                    ->click('.btn-close')
                    ->waitUntilMissing('#analyticsModal.show', 5)
                    ->assertNotPresent('#analyticsModal.show');
        });
    }
}
