<?php

namespace Tests\Browser;

use Laravel\Dusk\Browser;
use Tests\DuskTestCase;

class PublicacaoPaginacaoTest extends DuskTestCase
{
    /**
     * Teste: Login e verificação do elemento de paginação
     */
    public function testVerificarElementoPaginacao()
    {
        $this->browse(function (Browser $browser) {
            // Fazer login com credenciais reais
            $browser->visit('/login')
                    ->type('email', 'linuxewireless@gmail.com')
                    ->type('password', 'password')
                    ->press('Entrar')
                    ->waitForLocation('/dashboard', 10)
                    ->assertPathIs('/dashboard');

            // Navegar para publicações
            $browser->visit('/publicacoes')
                    ->waitFor('table', 10)
                    ->assertSee('Lista de Publicações');

            // Capturar screenshot inicial
            $browser->screenshot('paginacao-inicial');

            // Verificar se há elemento de paginação
            if ($browser->element('.pagination')) {
                $browser->screenshot('paginacao-encontrada');

                // Capturar HTML do elemento de paginação
                $paginacaoHtml = $browser->script("return document.querySelector('.pagination')?.outerHTML;");
                dump('HTML da Paginação:', $paginacaoHtml);

                // Verificar classes e estrutura
                $temLinks = $browser->script("return document.querySelectorAll('.pagination .page-link').length;");
                dump('Quantidade de links de página:', $temLinks);

                // Verificar se os links são clicáveis
                $linksVisiveis = $browser->script("
                    return Array.from(document.querySelectorAll('.pagination .page-link'))
                        .map(link => ({
                            text: link.textContent.trim(),
                            href: link.getAttribute('href'),
                            visible: link.offsetParent !== null,
                            disabled: link.closest('.page-item')?.classList.contains('disabled')
                        }));
                ");
                dump('Links de paginação:', $linksVisiveis);

                // Verificar estilos aplicados
                $estilos = $browser->script("
                    const pagination = document.querySelector('.pagination');
                    if (pagination) {
                        const computed = window.getComputedStyle(pagination);
                        return {
                            display: computed.display,
                            visibility: computed.visibility,
                            opacity: computed.opacity,
                            color: computed.color,
                            backgroundColor: computed.backgroundColor,
                            fontSize: computed.fontSize,
                            position: computed.position,
                            zIndex: computed.zIndex
                        };
                    }
                    return null;
                ");
                dump('Estilos CSS da paginação:', $estilos);

                // Verificar se há conflitos de CSS
                $conflitos = $browser->script("
                    const pagination = document.querySelector('.pagination');
                    const issues = [];

                    if (pagination) {
                        const computed = window.getComputedStyle(pagination);

                        // Verificar se está invisível
                        if (computed.display === 'none') issues.push('display: none');
                        if (computed.visibility === 'hidden') issues.push('visibility: hidden');
                        if (computed.opacity === '0') issues.push('opacity: 0');

                        // Verificar se cor do texto e fundo são iguais (invisível)
                        if (computed.color === computed.backgroundColor) {
                            issues.push('Cor do texto igual ao fundo');
                        }

                        // Verificar z-index negativo
                        if (parseInt(computed.zIndex) < 0) {
                            issues.push('z-index negativo: ' + computed.zIndex);
                        }
                    }

                    return issues;
                ");
                dump('Problemas detectados:', $conflitos);

                // Tentar clicar no próximo link se existir
                try {
                    $proximoLink = $browser->element('.pagination .page-link[rel="next"]');
                    if ($proximoLink) {
                        $browser->screenshot('antes-clicar-proximo');
                        $browser->click('.pagination .page-link[rel="next"]')
                                ->pause(2000)
                                ->screenshot('depois-clicar-proximo');

                        dump('Conseguiu clicar no link "Próximo"');
                    } else {
                        dump('Link "Próximo" não encontrado');
                    }
                } catch (\Exception $e) {
                    dump('Erro ao tentar clicar no próximo:', $e->getMessage());
                }

            } else {
                $browser->screenshot('paginacao-nao-encontrada');
                dump('Elemento de paginação NÃO encontrado no DOM');
            }

            // Capturar informações sobre quantas publicações existem
            $totalPublicacoes = $browser->script("
                return document.querySelectorAll('table tbody tr').length;
            ");
            dump('Total de linhas na tabela:', $totalPublicacoes);

            // Verificar se existe mensagem de paginação
            $textoPaginacao = $browser->script("
                const pag = document.querySelector('.pagination');
                return pag ? pag.parentElement.textContent : 'Paginação não encontrada';
            ");
            dump('Texto ao redor da paginação:', $textoPaginacao);
        });
    }

    /**
     * Teste: Verificar estrutura HTML completa da página
     */
    public function testVerificarEstruturaCompletaDaPagina()
    {
        $this->browse(function (Browser $browser) {
            // Login
            $browser->visit('/login')
                    ->type('email', 'linuxewireless@gmail.com')
                    ->type('password', 'password')
                    ->press('Entrar')
                    ->waitForLocation('/dashboard', 10);

            // Ir para publicações
            $browser->visit('/publicacoes')
                    ->waitFor('table', 10);

            // Capturar toda a estrutura da página
            $estrutura = $browser->script("
                return {
                    url: window.location.href,
                    title: document.title,
                    hasTable: !!document.querySelector('table'),
                    hasPagination: !!document.querySelector('.pagination'),
                    paginationLocation: document.querySelector('.pagination')?.parentElement?.className,
                    cardBody: !!document.querySelector('.card-body'),
                    totalCards: document.querySelectorAll('.card').length,
                    bodyHTML: document.querySelector('.card-body')?.innerHTML.substring(0, 500)
                };
            ");

            dump('Estrutura da página:', $estrutura);

            // Screenshot da página inteira
            $browser->screenshot('pagina-completa-publicacoes');

            // Verificar se @if($publicacoes->hasPages()) está funcionando
            $temPaginas = $browser->script("
                // Tentar encontrar qualquer elemento relacionado a paginação
                const selectors = [
                    '.pagination',
                    '.paginate',
                    '.page-link',
                    '[class*=\"paginat\"]',
                    '[class*=\"page\"]'
                ];

                const found = {};
                selectors.forEach(sel => {
                    const elements = document.querySelectorAll(sel);
                    if (elements.length > 0) {
                        found[sel] = elements.length;
                    }
                });

                return found;
            ");

            dump('Seletores de paginação encontrados:', $temPaginas);
        });
    }

    /**
     * Teste: Verificar console de erros JavaScript
     */
    public function testVerificarErrosJavaScript()
    {
        $this->browse(function (Browser $browser) {
            // Habilitar captura de console logs
            $browser->driver->manage()->logs();

            // Login
            $browser->visit('/login')
                    ->type('email', 'linuxewireless@gmail.com')
                    ->type('password', 'password')
                    ->press('Entrar')
                    ->waitForLocation('/dashboard', 10);

            // Ir para publicações
            $browser->visit('/publicacoes')
                    ->waitFor('table', 10)
                    ->pause(2000); // Aguardar carregamento completo

            // Capturar erros de JavaScript
            $errosJS = $browser->script("
                // Retornar erros capturados (se houver listener)
                return window.jsErrors || [];
            ");

            if (!empty($errosJS)) {
                dump('Erros JavaScript encontrados:', $errosJS);
            } else {
                dump('Nenhum erro JavaScript detectado');
            }

            // Screenshot final
            $browser->screenshot('teste-erros-js');
        });
    }

    /**
     * Teste: Inspecionar CSS carregado
     */
    public function testInspecionarCSSCarregado()
    {
        $this->browse(function (Browser $browser) {
            // Login
            $browser->visit('/login')
                    ->type('email', 'linuxewireless@gmail.com')
                    ->type('password', 'password')
                    ->press('Entrar')
                    ->waitForLocation('/dashboard', 10);

            // Ir para publicações
            $browser->visit('/publicacoes')
                    ->waitFor('table', 10);

            // Listar todos os arquivos CSS carregados
            $cssFiles = $browser->script("
                return Array.from(document.querySelectorAll('link[rel=\"stylesheet\"]'))
                    .map(link => link.href);
            ");

            dump('Arquivos CSS carregados:', $cssFiles);

            // Verificar se Bootstrap está carregado
            $bootstrapCarregado = $browser->script("
                return typeof bootstrap !== 'undefined' ||
                       document.querySelector('link[href*=\"bootstrap\"]') !== null;
            ");

            dump('Bootstrap carregado:', $bootstrapCarregado ? 'Sim' : 'Não');

            // Verificar classes do Bootstrap na paginação
            $classesPaginacao = $browser->script("
                const pag = document.querySelector('.pagination');
                if (pag) {
                    return {
                        classes: pag.className,
                        items: Array.from(pag.querySelectorAll('.page-item')).map(item => item.className),
                        links: Array.from(pag.querySelectorAll('.page-link')).map(link => link.className)
                    };
                }
                return null;
            ");

            dump('Classes Bootstrap na paginação:', $classesPaginacao);
        });
    }
}
