<?php

namespace Tests\Feature\Api;

use App\Models\HotspotCustomization;
use App\Models\Hotspot;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class CustomizationControllerTest extends TestCase
{
    use RefreshDatabase;

    /**
     * Testa endpoint /api/customization/{serial}/script.js com customização existente
     */
    public function test_script_endpoint_returns_javascript_for_existing_customization(): void
    {
        // Criar hotspot com serial
        $hotspot = Hotspot::factory()->create(['serial' => 'TEST-SERIAL-001']);

        // Criar customização ativa
        HotspotCustomization::factory()->active()->create([
            'hotspot_id' => $hotspot->id,
            'brand_name' => 'Test WiFi Brand',
            'primary_color' => '#FF0000',
        ]);

        // Request
        $response = $this->get('/api/customization/TEST-SERIAL-001/script.js');

        // Assertions
        $response->assertStatus(200);
        $response->assertHeader('Content-Type', 'application/javascript; charset=UTF-8');
        $response->assertHeader('Cache-Control', 'no-cache, no-store, must-revalidate');

        $content = $response->getContent();

        // Deve conter código JavaScript
        $this->assertStringContainsString('<script>', $content);
        $this->assertStringContainsString('window.freefiCustomization', $content);
        $this->assertStringContainsString('Test WiFi Brand', $content);
        $this->assertStringContainsString('#FF0000', $content);
    }

    /**
     * Testa endpoint script.js quando não há customização
     */
    public function test_script_endpoint_returns_empty_comment_when_no_customization(): void
    {
        $response = $this->get('/api/customization/NONEXISTENT-SERIAL/script.js');

        $response->assertStatus(200);
        $response->assertHeader('Content-Type', 'application/javascript; charset=UTF-8');

        $content = $response->getContent();
        $this->assertStringContainsString('// Nenhuma customização ativa', $content);
    }

    /**
     * Testa endpoint script.js ignora customizações inativas
     */
    public function test_script_endpoint_ignores_inactive_customizations(): void
    {
        $hotspot = Hotspot::factory()->create(['serial' => 'INACTIVE-TEST']);

        // Customização inativa
        HotspotCustomization::factory()->inactive()->create([
            'hotspot_id' => $hotspot->id,
        ]);

        $response = $this->get('/api/customization/INACTIVE-TEST/script.js');

        $response->assertStatus(200);
        $this->assertStringContainsString('// Nenhuma customização ativa', $response->getContent());
    }

    /**
     * Testa endpoint /api/customization/{serial} retorna JSON
     */
    public function test_show_endpoint_returns_json_for_existing_customization(): void
    {
        $hotspot = Hotspot::factory()->create(['serial' => 'JSON-TEST-001']);

        $customization = HotspotCustomization::factory()->active()->create([
            'hotspot_id' => $hotspot->id,
            'brand_name' => 'JSON Test Brand',
            'logo_url' => 'img/test-logo.png',
            'primary_color' => '#123456',
        ]);

        $response = $this->get('/api/customization/JSON-TEST-001');

        $response->assertStatus(200);
        $response->assertJson([
            'success' => true,
        ]);

        $data = $response->json('customization');
        $this->assertEquals('JSON Test Brand', $data['brand_name']);
        $this->assertEquals('img/test-logo.png', $data['logo_url']);
        $this->assertEquals('#123456', $data['primary_color']);
    }

    /**
     * Testa endpoint JSON quando não há customização
     */
    public function test_show_endpoint_returns_404_when_no_customization(): void
    {
        $response = $this->get('/api/customization/NONEXISTENT');

        $response->assertStatus(404);
        $response->assertJson([
            'success' => false,
            'message' => 'Nenhuma customização encontrada',
            'use_default' => true,
        ]);
    }

    /**
     * Testa que endpoints não requerem autenticação
     */
    public function test_endpoints_are_publicly_accessible(): void
    {
        $hotspot = Hotspot::factory()->create(['serial' => 'PUBLIC-TEST']);
        HotspotCustomization::factory()->active()->create(['hotspot_id' => $hotspot->id]);

        // Script endpoint
        $response1 = $this->get('/api/customization/PUBLIC-TEST/script.js');
        $response1->assertStatus(200);

        // JSON endpoint
        $response2 = $this->get('/api/customization/PUBLIC-TEST');
        $response2->assertStatus(200);
    }

    /**
     * Testa cache headers no endpoint script.js
     */
    public function test_script_endpoint_has_no_cache_headers(): void
    {
        $hotspot = Hotspot::factory()->create(['serial' => 'CACHE-TEST']);
        HotspotCustomization::factory()->active()->create(['hotspot_id' => $hotspot->id]);

        $response = $this->get('/api/customization/CACHE-TEST/script.js');

        $response->assertHeader('Cache-Control', 'no-cache, no-store, must-revalidate');
    }

    /**
     * Testa custom CSS/JS no script gerado
     */
    public function test_script_includes_custom_css_and_js(): void
    {
        $hotspot = Hotspot::factory()->create(['serial' => 'CUSTOM-CODE-TEST']);

        HotspotCustomization::factory()->create([
            'hotspot_id' => $hotspot->id,
            'custom_css' => '.my-custom-class { background: blue; }',
            'custom_js' => 'console.log("Custom JS loaded");',
            'is_active' => true,
        ]);

        $response = $this->get('/api/customization/CUSTOM-CODE-TEST/script.js');

        $content = $response->getContent();

        $this->assertStringContainsString('.my-custom-class { background: blue; }', $content);
        $this->assertStringContainsString('console.log("Custom JS loaded");', $content);
    }

    /**
     * Testa que retorna a customização mais recente se houver múltiplas
     */
    public function test_returns_latest_customization_if_multiple_exist(): void
    {
        $hotspot = Hotspot::factory()->create(['serial' => 'MULTIPLE-TEST']);

        // Criar primeira customização (mais antiga)
        $old = HotspotCustomization::factory()->active()->create([
            'hotspot_id' => $hotspot->id,
            'brand_name' => 'Old Brand',
        ]);

        // Deletar para poder criar outra (constraint unique)
        $old->forceDelete();

        // Criar segunda customização (mais recente)
        HotspotCustomization::factory()->active()->create([
            'hotspot_id' => $hotspot->id,
            'brand_name' => 'New Brand',
        ]);

        $response = $this->get('/api/customization/MULTIPLE-TEST');

        $response->assertStatus(200);
        $this->assertEquals('New Brand', $response->json('customization.brand_name'));
    }

    /**
     * Testa tratamento de caracteres especiais no serial
     */
    public function test_handles_special_characters_in_serial(): void
    {
        $hotspot = Hotspot::factory()->create(['serial' => 'TEST-SERIAL_123.456']);

        HotspotCustomization::factory()->active()->create(['hotspot_id' => $hotspot->id]);

        $response = $this->get('/api/customization/TEST-SERIAL_123.456/script.js');

        $response->assertStatus(200);
        $this->assertStringNotContainsString('// Nenhuma customização ativa', $response->getContent());
    }

    /**
     * Testa que todos os campos são incluídos no JSON
     */
    public function test_json_response_includes_all_fields(): void
    {
        $hotspot = Hotspot::factory()->create(['serial' => 'FULL-FIELDS-TEST']);

        HotspotCustomization::factory()->create([
            'hotspot_id' => $hotspot->id,
            'brand_name' => 'Full Test',
            'logo_url' => 'img/logo.png',
            'primary_color' => '#000000',
            'welcome_title' => 'Welcome!',
            'modal_countdown_seconds' => 15,
            'enable_advertisement' => true,
            'custom_css' => '.test {}',
            'is_active' => true,
        ]);

        $response = $this->get('/api/customization/FULL-FIELDS-TEST');

        $data = $response->json('customization');

        $this->assertArrayHasKey('brand_name', $data);
        $this->assertArrayHasKey('logo_url', $data);
        $this->assertArrayHasKey('primary_color', $data);
        $this->assertArrayHasKey('welcome_title', $data);
        $this->assertArrayHasKey('modal_countdown_seconds', $data);
        $this->assertArrayHasKey('enable_advertisement', $data);
        $this->assertArrayHasKey('custom_css', $data);
    }

    /**
     * Testa performance com múltiplas requisições
     */
    public function test_handles_multiple_concurrent_requests(): void
    {
        $hotspot = Hotspot::factory()->create(['serial' => 'PERFORMANCE-TEST']);
        HotspotCustomization::factory()->active()->create(['hotspot_id' => $hotspot->id]);

        // Simular 10 requisições
        for ($i = 0; $i < 10; $i++) {
            $response = $this->get('/api/customization/PERFORMANCE-TEST/script.js');
            $response->assertStatus(200);
        }

        $this->assertTrue(true); // Se chegou aqui, passou no teste de concorrência
    }
}
