<?php

use App\Models\Empresa;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;

uses(RefreshDatabase::class);

beforeEach(function () {
    // Criar dados de teste
    $this->matriz = Empresa::create([
        'name' => 'FreeFi Matriz',
        'email' => 'matriz@freefi.com',
        'tipo_empresa' => 'matriz',
        'status' => 0,
    ]);
    
    $this->franquiaAtiva = Empresa::create([
        'name' => 'FreeFi São Paulo',
        'email' => 'sp@freefi.com',
        'tipo_empresa' => 'franquia',
        'empresa_pai_id' => $this->matriz->id,
        'codigo_convite' => 'ABCD123456',
        'status' => 0, // ativa
    ]);
    
    $this->franquiaInativa = Empresa::create([
        'name' => 'FreeFi Rio',
        'email' => 'rj@freefi.com',
        'tipo_empresa' => 'franquia',
        'empresa_pai_id' => $this->matriz->id,
        'codigo_convite' => 'XPTO999999',
        'status' => 1, // inativa
    ]);
});

// ============================================
// TESTES DA TELA DE REGISTRO
// ============================================

test('tela de registro pode ser acessada', function () {
    $response = $this->get(route('register'));
    
    $response->assertStatus(200);
    $response->assertSee('Código de Convite');
});

test('usuário logado não pode acessar tela de registro', function () {
    $user = User::factory()->create();
    
    $response = $this->actingAs($user)->get(route('register'));
    
    $response->assertRedirect(route('dashboard'));
});

// ============================================
// TESTES DE VALIDAÇÃO DE CÓDIGO (AJAX)
// ============================================

test('validação aceita código válido de franquia ativa', function () {
    $response = $this->postJson(route('register.validate-code'), [
        'codigo' => 'ABCD123456'
    ]);
    
    $response->assertStatus(200)
             ->assertJson([
                 'valid' => true,
                 'empresa' => [
                     'id' => $this->franquiaAtiva->id,
                     'name' => 'FreeFi São Paulo',
                 ]
             ]);
});

test('validação rejeita código inexistente', function () {
    $response = $this->postJson(route('register.validate-code'), [
        'codigo' => 'INVALIDO00'
    ]);
    
    $response->assertStatus(422)
             ->assertJson([
                 'valid' => false,
                 'message' => 'Código de convite inválido ou franquia desativada.'
             ]);
});

test('validação rejeita código de franquia inativa', function () {
    $response = $this->postJson(route('register.validate-code'), [
        'codigo' => 'XPTO999999' // franquia inativa
    ]);
    
    $response->assertStatus(422)
             ->assertJson([
                 'valid' => false,
             ]);
});

test('validação rejeita código com tamanho inválido', function () {
    $response = $this->postJson(route('register.validate-code'), [
        'codigo' => 'ABC123' // apenas 6 caracteres
    ]);
    
    $response->assertStatus(422)
             ->assertJsonValidationErrors(['codigo']);
});

test('validação converte código para maiúsculo', function () {
    // Criar código em minúsculo
    $franquia = Empresa::create([
        'name' => 'Teste Lower',
        'email' => 'test@test.com',
        'tipo_empresa' => 'franquia',
        'empresa_pai_id' => $this->matriz->id,
        'codigo_convite' => 'LOWER12345',
        'status' => 0,
    ]);
    
    $response = $this->postJson(route('register.validate-code'), [
        'codigo' => 'lower12345' // minúsculo
    ]);
    
    $response->assertStatus(200)
             ->assertJson(['valid' => true]);
});

// ============================================
// TESTES DE REGISTRO COMPLETO
// ============================================

test('usuário pode se registrar com código válido', function () {
    $response = $this->post(route('register.store'), [
        'codigo_convite' => 'ABCD123456',
        'name' => 'João da Silva',
        'email' => 'joao@example.com',
        'password' => 'senha12345',
        'password_confirmation' => 'senha12345',
    ]);
    
    $response->assertRedirect(route('login'));
    $response->assertSessionHas('success');
    
    // Verificar que usuário foi criado
    $this->assertDatabaseHas('users', [
        'name' => 'João da Silva',
        'email' => 'joao@example.com',
        'empresa_id' => $this->franquiaAtiva->id,
        'status_cadastro' => 'pendente',
    ]);
    
    // Verificar senha criptografada
    $user = User::where('email', 'joao@example.com')->first();
    $this->assertTrue(\Hash::check('senha12345', $user->password));
});

test('registro rejeita código inválido', function () {
    $response = $this->post(route('register.store'), [
        'codigo_convite' => 'INVALIDO00',
        'name' => 'João da Silva',
        'email' => 'joao@example.com',
        'password' => 'senha12345',
        'password_confirmation' => 'senha12345',
    ]);
    
    $response->assertSessionHasErrors(['codigo_convite']);
    
    $this->assertDatabaseMissing('users', [
        'email' => 'joao@example.com'
    ]);
});

test('registro rejeita código de franquia inativa', function () {
    $response = $this->post(route('register.store'), [
        'codigo_convite' => 'XPTO999999', // inativa
        'name' => 'João da Silva',
        'email' => 'joao@example.com',
        'password' => 'senha12345',
        'password_confirmation' => 'senha12345',
    ]);
    
    $response->assertSessionHasErrors(['codigo_convite']);
});

test('registro rejeita email duplicado', function () {
    // Criar usuário existente
    User::factory()->create(['email' => 'existente@example.com']);
    
    $response = $this->post(route('register.store'), [
        'codigo_convite' => 'ABCD123456',
        'name' => 'João da Silva',
        'email' => 'existente@example.com', // email duplicado
        'password' => 'senha12345',
        'password_confirmation' => 'senha12345',
    ]);
    
    $response->assertSessionHasErrors(['email']);
});

test('registro rejeita nome muito curto', function () {
    $response = $this->post(route('register.store'), [
        'codigo_convite' => 'ABCD123456',
        'name' => 'AB', // apenas 2 caracteres
        'email' => 'joao@example.com',
        'password' => 'senha12345',
        'password_confirmation' => 'senha12345',
    ]);
    
    $response->assertSessionHasErrors(['name']);
});

test('registro rejeita senha menor que 8 caracteres', function () {
    $response = $this->post(route('register.store'), [
        'codigo_convite' => 'ABCD123456',
        'name' => 'João da Silva',
        'email' => 'joao@example.com',
        'password' => '1234567', // apenas 7 caracteres
        'password_confirmation' => '1234567',
    ]);
    
    $response->assertSessionHasErrors(['password']);
});

test('registro rejeita senhas que não coincidem', function () {
    $response = $this->post(route('register.store'), [
        'codigo_convite' => 'ABCD123456',
        'name' => 'João da Silva',
        'email' => 'joao@example.com',
        'password' => 'senha12345',
        'password_confirmation' => 'senhadiferente', // não coincide
    ]);
    
    $response->assertSessionHasErrors(['password']);
});

test('registro rejeita email inválido', function () {
    $response = $this->post(route('register.store'), [
        'codigo_convite' => 'ABCD123456',
        'name' => 'João da Silva',
        'email' => 'email-invalido', // sem @
        'password' => 'senha12345',
        'password_confirmation' => 'senha12345',
    ]);
    
    $response->assertSessionHasErrors(['email']);
});

test('telefone é opcional no registro', function () {
    $response = $this->post(route('register.store'), [
        'codigo_convite' => 'ABCD123456',
        'name' => 'João da Silva',
        'email' => 'joao@example.com',
        'password' => 'senha12345',
        'password_confirmation' => 'senha12345',
        // phone não fornecido
    ]);
    
    $response->assertRedirect(route('login'));
    
    $this->assertDatabaseHas('users', [
        'email' => 'joao@example.com',
        'phone' => null,
    ]);
});

test('usuário criado tem status pendente', function () {
    $this->post(route('register.store'), [
        'codigo_convite' => 'ABCD123456',
        'name' => 'João da Silva',
        'email' => 'joao@example.com',
        'password' => 'senha12345',
        'password_confirmation' => 'senha12345',
    ]);
    
    $user = User::where('email', 'joao@example.com')->first();
    
    expect($user->status_cadastro)->toBe('pendente');
    expect($user->aprovado_por)->toBeNull();
    expect($user->aprovado_em)->toBeNull();
});

test('campos obrigatórios são validados', function () {
    $response = $this->post(route('register.store'), []);
    
    $response->assertSessionHasErrors([
        'codigo_convite',
        'name',
        'email',
        'password',
    ]);
});

test('código é convertido para maiúsculo no registro', function () {
    $this->post(route('register.store'), [
        'codigo_convite' => 'abcd123456', // minúsculo
        'name' => 'João da Silva',
        'email' => 'joao@example.com',
        'password' => 'senha12345',
        'password_confirmation' => 'senha12345',
    ]);
    
    // Deve encontrar a franquia mesmo com código em minúsculo
    $this->assertDatabaseHas('users', [
        'email' => 'joao@example.com',
        'empresa_id' => $this->franquiaAtiva->id,
    ]);
});

// ============================================
// TESTES DE SEGURANÇA
// ============================================

test('registro usa transação para garantir integridade', function () {
    // Simular erro forçando email duplicado após validação
    User::factory()->create(['email' => 'test@test.com']);
    
    // Tentar registrar com mesmo email (deve falhar na inserção)
    try {
        $this->withoutExceptionHandling()
             ->post(route('register.store'), [
                 'codigo_convite' => 'ABCD123456',
                 'name' => 'Teste',
                 'email' => 'test@test.com',
                 'password' => 'senha12345',
                 'password_confirmation' => 'senha12345',
             ]);
    } catch (\Exception $e) {
        // Esperado
    }
    
    // Verificar que não há duplicatas
    expect(User::where('email', 'test@test.com')->count())->toBe(1);
});

test('senha nunca é armazenada em texto plano', function () {
    $this->post(route('register.store'), [
        'codigo_convite' => 'ABCD123456',
        'name' => 'João da Silva',
        'email' => 'joao@example.com',
        'password' => 'senha12345',
        'password_confirmation' => 'senha12345',
    ]);
    
    $user = User::where('email', 'joao@example.com')->first();
    
    // Senha não pode ser igual ao texto plano
    expect($user->password)->not->toBe('senha12345');
    
    // Senha deve começar com $2y$ (bcrypt)
    expect($user->password)->toStartWith('$2y$');
});
