<?php

namespace Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use App\Models\User;
use App\Models\Empresa;

class EmpresaCrudTest extends TestCase
{
    use RefreshDatabase;

    protected $user;

    protected function setUp(): void
    {
        parent::setUp();
        $this->user = User::factory()->create([
            'is_superadmin' => true,
            'status_cadastro' => 'aprovado',
        ]);
        $this->actingAs($this->user);
    }

    public function test_can_view_empresas_index()
    {
        $response = $this->get(route('empresas.index'));
        $response->assertStatus(200);
        $response->assertViewIs('empresas.index');
    }

    public function test_can_view_empresa_create_form()
    {
        $response = $this->get(route('empresas.create'));
        $response->assertStatus(200);
        $response->assertViewIs('empresas.create');
    }

    public function test_can_create_empresa()
    {
        $empresaData = [
            'name' => 'New Empresa',
            'email' => 'empresa@example.com',
            'phone' => '123456789',
        ];

        $response = $this->post(route('empresas.store'), $empresaData);

        $response->assertRedirect(route('empresas.index'));
        $this->assertDatabaseHas('empresas', ['name' => 'New Empresa']);
    }

    public function test_can_view_empresa_edit_form()
    {
        $empresa = Empresa::factory()->create();
        $response = $this->get(route('empresas.edit', $empresa));
        $response->assertStatus(200);
        $response->assertViewIs('empresas.edit');
    }

    public function test_can_update_empresa()
    {
        $empresa = Empresa::factory()->create();
        $updatedData = [
            'name' => 'Updated Empresa Name',
            'email' => 'updatedempresa@example.com',
        ];

        $response = $this->put(route('empresas.update', $empresa), $updatedData);

        $response->assertRedirect(route('empresas.index'));
        $this->assertDatabaseHas('empresas', ['id' => $empresa->id, 'name' => 'Updated Empresa Name']);
    }

    public function test_can_delete_empresa()
    {
        $empresa = Empresa::factory()->create();
        $response = $this->delete(route('empresas.destroy', $empresa));

        $response->assertRedirect(route('empresas.index'));
        $this->assertDatabaseMissing('empresas', ['id' => $empresa->id]);
    }
}
