<?php

use App\Models\Empresa;
use App\Models\User;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;

test('can upload company logo on create', function () {
    $user = User::factory()->create();
    $this->actingAs($user);
    
    Storage::fake('public');
    
    $logo = UploadedFile::fake()->image('logo.jpg', 400, 300);
    
    $response = $this->post(route('empresas.store'), [
        'name' => 'Empresa Teste',
        'email' => 'teste@empresa.com',
        'phone' => '(11) 99999-9999',
        'logomarca' => $logo
    ]);
    
    $response->assertRedirect(route('empresas.index'));
    
    $empresa = Empresa::where('name', 'Empresa Teste')->first();
    $this->assertNotNull($empresa);
    $this->assertNotNull($empresa->logomarca);
    
    // Verificar se o arquivo foi salvo
    $this->assertTrue(Storage::disk('public')->exists($empresa->logomarca));
});

test('can update company logo', function () {
    $user = User::factory()->create();
    $this->actingAs($user);
    
    Storage::fake('public');
    
    $empresa = Empresa::factory()->create([
        'logomarca' => 'logos/old_logo.jpg'
    ]);
    
    // Criar arquivo antigo
    Storage::disk('public')->put('logos/old_logo.jpg', 'old content');
    
    $newLogo = UploadedFile::fake()->image('new_logo.jpg', 500, 400);
    
    $response = $this->put(route('empresas.update', $empresa), [
        'name' => $empresa->name,
        'email' => $empresa->email,
        'phone' => $empresa->phone,
        'logomarca' => $newLogo
    ]);
    
    $response->assertRedirect(route('empresas.index'));
    
    $empresa->refresh();
    $this->assertNotEquals('logos/old_logo.jpg', $empresa->logomarca);
    
    // Verificar se o arquivo antigo foi deletado
    $this->assertFalse(Storage::disk('public')->exists('logos/old_logo.jpg'));
    
    // Verificar se o novo arquivo foi salvo
    $this->assertTrue(Storage::disk('public')->exists($empresa->logomarca));
});

test('validates logo file type and size', function () {
    $user = User::factory()->create();
    $this->actingAs($user);
    
    Storage::fake('public');
    
    // Teste com arquivo muito grande (3MB)
    $largeLogo = UploadedFile::fake()->create('logo.jpg', 3000);
    
    $response = $this->post(route('empresas.store'), [
        'name' => 'Empresa Teste',
        'email' => 'teste@empresa.com',
        'phone' => '(11) 99999-9999',
        'logomarca' => $largeLogo
    ]);
    
    $response->assertSessionHasErrors(['logomarca']);
});

test('can delete company with logo', function () {
    $user = User::factory()->create();
    $this->actingAs($user);
    
    Storage::fake('public');
    
    $empresa = Empresa::factory()->create([
        'logomarca' => 'logos/logo_to_delete.jpg'
    ]);
    
    // Criar arquivo
    Storage::disk('public')->put('logos/logo_to_delete.jpg', 'content');
    
    $response = $this->delete(route('empresas.destroy', $empresa));
    
    $response->assertRedirect(route('empresas.index'));
    
    // Verificar se o arquivo foi deletado
    $this->assertFalse(Storage::disk('public')->exists('logos/logo_to_delete.jpg'));
    
    // Verificar se a empresa foi soft deleted
    $this->assertNull(Empresa::find($empresa->id));
    $this->assertNotNull(Empresa::withTrashed()->find($empresa->id));
});