<?php

use App\Models\User;
use App\Models\Empresa;
use App\Models\Hotspot;

beforeEach(function () {
    $this->user = User::factory()->create([
        'is_superadmin' => true,
        'status_cadastro' => 'aprovado',
    ]);
    $this->actingAs($this->user);
    $this->empresa = Empresa::factory()->create();
});

it('can display hotspots index page', function () {
    $response = $this->get(route('hotspots.index'));

    $response->assertStatus(200);
    $response->assertViewIs('hotspots.index');
    $response->assertViewHas('hotspots');
});

it('can search hotspots', function () {
    $hotspot1 = Hotspot::factory()->create([
        'marca' => 'Mikrotik',
        'modelo' => 'hAP ac2',
        'serial' => 'AA1234BB5678',
        'empresa_id' => $this->empresa->id,
    ]);

    $hotspot2 = Hotspot::factory()->create([
        'marca' => 'Ubiquiti',
        'modelo' => 'UniFi AP AC Lite',
        'serial' => 'CC9876DD4321',
        'empresa_id' => $this->empresa->id,
    ]);

    $response = $this->get(route('hotspots.index', ['search' => 'Mikrotik']));

    $response->assertStatus(200);
    $response->assertSee('Mikrotik');
    $response->assertDontSee('Ubiquiti');
});

it('can display hotspots create page', function () {
    $response = $this->get(route('hotspots.create'));

    $response->assertStatus(200);
    $response->assertViewIs('hotspots.create');
    $response->assertViewHas('empresas');
});

it('can create a new hotspot', function () {
    $hotspotData = [
        'marca' => 'Mikrotik',
        'modelo' => 'hAP ac2',
        'serial' => 'XX1234YY5678',
        'empresa_id' => $this->empresa->id,
    ];

    $response = $this->post(route('hotspots.store'), $hotspotData);

    $response->assertRedirect(route('hotspots.index'));
    $response->assertSessionHas('success');

    $this->assertDatabaseHas('hotspots', [
        'marca' => 'Mikrotik',
        'modelo' => 'hAP ac2',
        'serial' => 'XX1234YY5678',
        'empresa_id' => $this->empresa->id,
    ]);
});

it('validates required fields when creating a hotspot', function () {
    $response = $this->post(route('hotspots.store'), []);

    $response->assertSessionHasErrors(['marca', 'modelo', 'serial', 'empresa_id']);
});

it('validates unique serial when creating a hotspot', function () {
    $existingHotspot = Hotspot::factory()->create([
        'serial' => 'DUPLICATE123',
        'empresa_id' => $this->empresa->id,
    ]);

    $response = $this->post(route('hotspots.store'), [
        'marca' => 'Mikrotik',
        'modelo' => 'hAP ac2',
        'serial' => 'DUPLICATE123',
        'empresa_id' => $this->empresa->id,
    ]);

    $response->assertSessionHasErrors(['serial']);
});

it('can display hotspot edit page', function () {
    $hotspot = Hotspot::factory()->create([
        'empresa_id' => $this->empresa->id,
    ]);

    $response = $this->get(route('hotspots.edit', $hotspot));

    $response->assertStatus(200);
    $response->assertViewIs('hotspots.edit');
    $response->assertViewHas('hotspot', $hotspot);
    $response->assertViewHas('empresas');
});

it('can update a hotspot', function () {
    $hotspot = Hotspot::factory()->create([
        'marca' => 'Mikrotik',
        'modelo' => 'hAP ac2',
        'serial' => 'OLD123456789',
        'empresa_id' => $this->empresa->id,
    ]);

    $updatedData = [
        'marca' => 'Ubiquiti',
        'modelo' => 'UniFi AP AC Pro',
        'serial' => 'NEW987654321',
        'empresa_id' => $this->empresa->id,
    ];

    $response = $this->put(route('hotspots.update', $hotspot), $updatedData);

    $response->assertRedirect(route('hotspots.index'));
    $response->assertSessionHas('success');

    $this->assertDatabaseHas('hotspots', [
        'id' => $hotspot->id,
        'marca' => 'Ubiquiti',
        'modelo' => 'UniFi AP AC Pro',
        'serial' => 'NEW987654321',
    ]);
});

it('can update hotspot without changing serial', function () {
    $hotspot = Hotspot::factory()->create([
        'marca' => 'Mikrotik',
        'modelo' => 'hAP ac2',
        'serial' => 'SAME123456789',
        'empresa_id' => $this->empresa->id,
    ]);

    $updatedData = [
        'marca' => 'Mikrotik',
        'modelo' => 'hAP ac3',
        'serial' => 'SAME123456789',
        'empresa_id' => $this->empresa->id,
    ];

    $response = $this->put(route('hotspots.update', $hotspot), $updatedData);

    $response->assertRedirect(route('hotspots.index'));
    $response->assertSessionHas('success');
});

it('validates unique serial when updating a hotspot', function () {
    $hotspot1 = Hotspot::factory()->create([
        'serial' => 'FIRST123456',
        'empresa_id' => $this->empresa->id,
    ]);

    $hotspot2 = Hotspot::factory()->create([
        'serial' => 'SECOND789012',
        'empresa_id' => $this->empresa->id,
    ]);

    $response = $this->put(route('hotspots.update', $hotspot2), [
        'marca' => 'Mikrotik',
        'modelo' => 'hAP ac2',
        'serial' => 'FIRST123456',
        'empresa_id' => $this->empresa->id,
    ]);

    $response->assertSessionHasErrors(['serial']);
});

it('can soft delete a hotspot', function () {
    $hotspot = Hotspot::factory()->create([
        'empresa_id' => $this->empresa->id,
    ]);

    $response = $this->delete(route('hotspots.destroy', $hotspot));

    $response->assertRedirect(route('hotspots.index'));
    $response->assertSessionHas('success');

    $this->assertSoftDeleted('hotspots', [
        'id' => $hotspot->id,
    ]);
});

it('can toggle hotspot status', function () {
    $hotspot = Hotspot::factory()->create([
        'empresa_id' => $this->empresa->id,
        'status' => true,
    ]);

    $response = $this->post(route('hotspots.toggleStatus', $hotspot));

    $response->assertRedirect(route('hotspots.index'));
    $response->assertSessionHas('success');

    $this->assertDatabaseHas('hotspots', [
        'id' => $hotspot->id,
        'status' => false,
    ]);

    // Toggle back
    $response = $this->post(route('hotspots.toggleStatus', $hotspot));

    $this->assertDatabaseHas('hotspots', [
        'id' => $hotspot->id,
        'status' => true,
    ]);
});

it('can display trash page with deleted hotspots', function () {
    $hotspot = Hotspot::factory()->create([
        'empresa_id' => $this->empresa->id,
    ]);

    $hotspot->delete();

    $response = $this->get(route('hotspots.trash'));

    $response->assertStatus(200);
    $response->assertViewIs('hotspots.trash');
    $response->assertViewHas('hotspots');
});

it('can restore a deleted hotspot', function () {
    $hotspot = Hotspot::factory()->create([
        'empresa_id' => $this->empresa->id,
    ]);

    $hotspot->delete();

    $response = $this->post(route('hotspots.restore', $hotspot->id));

    $response->assertRedirect(route('hotspots.trash'));
    $response->assertSessionHas('success');

    $this->assertDatabaseHas('hotspots', [
        'id' => $hotspot->id,
        'deleted_at' => null,
    ]);
});

it('has correct relationship with empresa', function () {
    $hotspot = Hotspot::factory()->create([
        'empresa_id' => $this->empresa->id,
    ]);

    expect($hotspot->empresa)->toBeInstanceOf(Empresa::class);
    expect($hotspot->empresa->id)->toBe($this->empresa->id);
});

it('empresa has correct relationship with hotspots', function () {
    $hotspot1 = Hotspot::factory()->create([
        'empresa_id' => $this->empresa->id,
    ]);

    $hotspot2 = Hotspot::factory()->create([
        'empresa_id' => $this->empresa->id,
    ]);

    $this->empresa->refresh();

    expect($this->empresa->hotspots)->toHaveCount(2);
    expect($this->empresa->hotspots->first())->toBeInstanceOf(Hotspot::class);
});

it('tracks user who created hotspot', function () {
    $hotspot = Hotspot::factory()->create([
        'empresa_id' => $this->empresa->id,
        'created_by' => $this->user->id,
    ]);

    expect($hotspot->creator)->toBeInstanceOf(User::class);
    expect($hotspot->creator->id)->toBe($this->user->id);
});

it('tracks user who updated hotspot', function () {
    $hotspot = Hotspot::factory()->create([
        'empresa_id' => $this->empresa->id,
        'updated_by' => $this->user->id,
    ]);

    expect($hotspot->updater)->toBeInstanceOf(User::class);
    expect($hotspot->updater->id)->toBe($this->user->id);
});

it('paginates hotspots correctly', function () {
    Hotspot::factory()->count(25)->create([
        'empresa_id' => $this->empresa->id,
    ]);

    $response = $this->get(route('hotspots.index'));

    $response->assertStatus(200);
    $response->assertViewHas('hotspots');
    
    $hotspots = $response->viewData('hotspots');
    expect($hotspots->perPage())->toBe(10);
    expect($hotspots->total())->toBe(25);
});
