<?php

namespace Tests\Feature\Publicacao;

use Tests\TestCase;
use App\Models\{Publicacao, User, Empresa, Criativo, Hotspot};
use App\Models\{PublicacaoImpressao, PublicacaoClique};
use Illuminate\Foundation\Testing\RefreshDatabase;
use Carbon\Carbon;

/**
 * Testes do Modal de Analytics
 * 
 * Testa se o endpoint analytics-json retorna os dados corretos
 * para o modal de visualizações
 */
class PublicacaoAnalyticsModalTest extends TestCase
{
    use RefreshDatabase;

    protected $user;
    protected $publicacao;
    protected $hotspot;

    protected function setUp(): void
    {
        parent::setUp();

        $this->withoutMiddleware();

        $this->user = User::factory()->create();
        
        $franquia = Empresa::factory()->create(['tipo_empresa' => 'franquia']);
        $cliente = Empresa::factory()->create(['tipo_empresa' => 'cliente']);
        $criativo = Criativo::factory()->create([
            'status_aprovacao' => 'aprovado',
            'status' => 0
        ]);

        $this->hotspot = Hotspot::factory()->create([
            'franquia_id' => $franquia->id,
            'serial' => 'TEST123',
        ]);

        $this->publicacao = Publicacao::factory()->create([
            'franquia_id' => $franquia->id,
            'cliente_id' => $cliente->id,
            'criativo_id' => $criativo->id,
            'titulo' => 'Publicação Teste Analytics',
            'track_impressoes' => true,
            'track_cliques' => true,
            'total_visualizacoes' => 100,
        ]);
    }

    // ==========================================
    // TESTE 1: Endpoint retorna 200
    // ==========================================
    public function test_endpoint_analytics_json_retorna_200()
    {
        $response = $this->actingAs($this->user)
            ->get(route('publicacoes.analyticsJson', $this->publicacao));

        $response->assertStatus(200);
    }

    // ==========================================
    // TESTE 2: Retorna JSON válido
    // ==========================================
    public function test_endpoint_retorna_json_valido()
    {
        $response = $this->actingAs($this->user)
            ->get(route('publicacoes.analyticsJson', $this->publicacao));

        $response->assertJsonStructure([
            'publicacao' => [
                'id',
                'titulo',
                'status',
                'data_inicio',
                'data_fim',
            ],
            'stats' => [
                'total_visualizacoes',
                'total_impressoes',
                'total_usuarios_unicos',
                'total_cliques',
                'ctr',
                'media_diaria_impressoes',
                'media_diaria_cliques',
                'usando_dados_detalhados',
            ],
        ]);
    }

    // ==========================================
    // TESTE 3: Dados da publicação corretos
    // ==========================================
    public function test_dados_da_publicacao_estao_corretos()
    {
        $response = $this->actingAs($this->user)
            ->get(route('publicacoes.analyticsJson', $this->publicacao));

        $json = $response->json();

        $this->assertEquals($this->publicacao->id, $json['publicacao']['id']);
        $this->assertEquals('Publicação Teste Analytics', $json['publicacao']['titulo']);
        $this->assertEquals('ativa', $json['publicacao']['status']);
    }

    // ==========================================
    // TESTE 4: Estatísticas básicas sem dados
    // ==========================================
    public function test_estatisticas_basicas_sem_dados_detalhados()
    {
        $response = $this->actingAs($this->user)
            ->get(route('publicacoes.analyticsJson', $this->publicacao));

        $json = $response->json();

        // Deve usar total_visualizacoes como fallback
        $this->assertEquals(100, $json['stats']['total_visualizacoes']);
        $this->assertEquals(0, $json['stats']['total_usuarios_unicos']);
        $this->assertEquals(0, $json['stats']['total_cliques']);
        $this->assertEquals(0, $json['stats']['ctr']);
        $this->assertFalse($json['stats']['usando_dados_detalhados']);
    }

    // ==========================================
    // TESTE 5: Impressões detalhadas
    // ==========================================
    public function test_com_impressoes_detalhadas()
    {
        // Criar impressões para hoje
        PublicacaoImpressao::factory()->create([
            'publicacao_id' => $this->publicacao->id,
            'hotspot_id' => $this->hotspot->id,
            'criativo_id' => $this->publicacao->criativo_id,
            'data' => Carbon::today(),
            'impressoes' => 50,
            'usuarios_unicos' => 30,
        ]);

        PublicacaoImpressao::factory()->create([
            'publicacao_id' => $this->publicacao->id,
            'hotspot_id' => $this->hotspot->id,
            'criativo_id' => $this->publicacao->criativo_id,
            'data' => Carbon::today(),
            'impressoes' => 30,
            'usuarios_unicos' => 20,
        ]);

        $response = $this->actingAs($this->user)
            ->get(route('publicacoes.analyticsJson', $this->publicacao));

        $json = $response->json();

        // Deve somar as impressões
        $this->assertEquals(80, $json['stats']['total_impressoes']);
        $this->assertEquals(50, $json['stats']['total_usuarios_unicos']);
        $this->assertTrue($json['stats']['usando_dados_detalhados']);
    }

    // ==========================================
    // TESTE 6: Cliques registrados
    // ==========================================
    public function test_cliques_registrados_corretamente()
    {
        // Criar impressões
        PublicacaoImpressao::factory()->create([
            'publicacao_id' => $this->publicacao->id,
            'hotspot_id' => $this->hotspot->id,
            'criativo_id' => $this->publicacao->criativo_id,
            'data' => Carbon::today(),
            'impressoes' => 100,
            'usuarios_unicos' => 50,
        ]);

        // Criar cliques
        PublicacaoClique::factory()->count(10)->create([
            'publicacao_id' => $this->publicacao->id,
            'hotspot_id' => $this->hotspot->id,
            'criativo_id' => $this->publicacao->criativo_id,
        ]);

        $response = $this->actingAs($this->user)
            ->get(route('publicacoes.analyticsJson', $this->publicacao));

        $json = $response->json();

        $this->assertEquals(10, $json['stats']['total_cliques']);
        $this->assertEquals(10.0, $json['stats']['ctr']); // 10/100 * 100 = 10%
    }

    // ==========================================
    // TESTE 7: CTR calculado corretamente
    // ==========================================
    public function test_ctr_calculado_corretamente()
    {
        // 200 impressões, 20 cliques = 10% CTR
        PublicacaoImpressao::factory()->create([
            'publicacao_id' => $this->publicacao->id,
            'hotspot_id' => $this->hotspot->id,
            'criativo_id' => $this->publicacao->criativo_id,
            'data' => Carbon::today(),
            'impressoes' => 200,
            'usuarios_unicos' => 100,
        ]);

        PublicacaoClique::factory()->count(20)->create([
            'publicacao_id' => $this->publicacao->id,
            'hotspot_id' => $this->hotspot->id,
            'criativo_id' => $this->publicacao->criativo_id,
        ]);

        $response = $this->actingAs($this->user)
            ->get(route('publicacoes.analyticsJson', $this->publicacao));

        $json = $response->json();

        $this->assertEquals(10.0, $json['stats']['ctr']);
    }

    // ==========================================
    // TESTE 8: Médias diárias calculadas
    // ==========================================
    public function test_medias_diarias_calculadas()
    {
        // Criar publicação de 10 dias atrás até hoje
        $publicacao = Publicacao::factory()->create([
            'franquia_id' => $this->publicacao->franquia_id,
            'cliente_id' => $this->publicacao->cliente_id,
            'criativo_id' => $this->publicacao->criativo_id,
            'data_inicio' => Carbon::now()->subDays(10),
            'data_fim' => Carbon::now()->addDays(10),
            'track_impressoes' => true,
            'track_cliques' => true,
        ]);

        // 100 impressões em 10 dias = 10 por dia
        PublicacaoImpressao::factory()->create([
            'publicacao_id' => $publicacao->id,
            'hotspot_id' => $this->hotspot->id,
            'criativo_id' => $publicacao->criativo_id,
            'data' => Carbon::today(),
            'impressoes' => 100,
            'usuarios_unicos' => 50,
        ]);

        // 10 cliques em 10 dias = 1 por dia
        PublicacaoClique::factory()->count(10)->create([
            'publicacao_id' => $publicacao->id,
            'hotspot_id' => $this->hotspot->id,
            'criativo_id' => $publicacao->criativo_id,
        ]);

        $response = $this->actingAs($this->user)
            ->get(route('publicacoes.analyticsJson', $publicacao));

        $json = $response->json();

        // Médias devem estar presentes
        $this->assertGreaterThan(0, $json['stats']['media_diaria_impressoes']);
        $this->assertGreaterThan(0, $json['stats']['media_diaria_cliques']);
    }

    // ==========================================
    // TESTE 9: Impressões recentes (últimos 7 dias)
    // ==========================================
    public function test_impressoes_recentes_ultimos_7_dias()
    {
        // Criar impressões nos últimos 7 dias
        for ($i = 0; $i < 7; $i++) {
            PublicacaoImpressao::factory()->create([
                'publicacao_id' => $this->publicacao->id,
                'hotspot_id' => $this->hotspot->id,
                'criativo_id' => $this->publicacao->criativo_id,
                'data' => Carbon::today()->subDays($i),
                'impressoes' => 10 * ($i + 1),
                'usuarios_unicos' => 5 * ($i + 1),
            ]);
        }

        $response = $this->actingAs($this->user)
            ->get(route('publicacoes.analyticsJson', $this->publicacao));

        $json = $response->json();

        // Deve ter dados de impressões recentes
        $this->assertArrayHasKey('impressoes_recentes', $json);
    }

    // ==========================================
    // TESTE 10: Logs de acesso presentes
    // ==========================================
    public function test_logs_acesso_presentes_no_retorno()
    {
        PublicacaoImpressao::factory()->create([
            'publicacao_id' => $this->publicacao->id,
            'hotspot_id' => $this->hotspot->id,
            'criativo_id' => $this->publicacao->criativo_id,
            'data' => Carbon::today(),
            'impressoes' => 50,
            'usuarios_unicos' => 30,
        ]);

        $response = $this->actingAs($this->user)
            ->get(route('publicacoes.analyticsJson', $this->publicacao));

        $json = $response->json();

        $this->assertArrayHasKey('logs_acesso', $json);
        $this->assertArrayHasKey('total_logs', $json);
    }

    // ==========================================
    // TESTE 11: Performance do endpoint
    // ==========================================
    public function test_endpoint_responde_rapido()
    {
        $inicio = microtime(true);

        $response = $this->actingAs($this->user)
            ->get(route('publicacoes.analyticsJson', $this->publicacao));

        $tempo = (microtime(true) - $inicio) * 1000; // em ms

        $response->assertStatus(200);
        
        // Deve responder em menos de 1 segundo
        $this->assertLessThan(1000, $tempo, 
            "Endpoint demorou {$tempo}ms (limite: 1000ms)"
        );
    }
}
