<?php

namespace Tests\Feature;

use App\Models\Publicacao;
use App\Models\AuditoriaSplash;
use App\Models\User;
use Carbon\Carbon;
use Tests\TestCase;

class PublicacaoEstatisticasFeatureTest extends TestCase
{
    protected $user;

    protected function setUp(): void
    {
        parent::setUp();
        $this->user = User::factory()->create();
    }

    /**
     * Teste: Página de estatísticas carrega com sucesso
     */
    public function test_estatisticas_page_loads_successfully()
    {
        $publicacao = Publicacao::factory()->create([
            'data_inicio' => Carbon::now()->subDays(10),
            'data_fim' => Carbon::now()->addDays(10),
        ]);

        $response = $this->actingAs($this->user)
            ->get(route('publicacoes.estatisticas', $publicacao));

        $response->assertStatus(200);
        $response->assertViewIs('publicacoes.estatisticas');
    }

    /**
     * Teste: Seção Top 10 Hotspots é exibida
     */
    public function test_top_10_hotspots_section_displays()
    {
        $publicacao = Publicacao::factory()->create([
            'data_inicio' => Carbon::now()->subDays(10),
            'data_fim' => Carbon::now()->addDays(10),
        ]);

        // Criar dados
        for ($i = 1; $i <= 5; $i++) {
            AuditoriaSplash::create([
                'hs_serial' => "HS{$i}",
                'cl_mac_address' => 'AA:BB:CC:DD:EE:0' . $i,
                'pag_click' => 'http://example.com',
                'created_at' => now(),
            ]);
        }

        $response = $this->actingAs($this->user)
            ->get(route('publicacoes.estatisticas', $publicacao));

        $response->assertViewHas('stats', function($stats) {
            return isset($stats['top_hotspots']) && count($stats['top_hotspots']) > 0;
        });
    }

    /**
     * Teste: Seção Dados por Dia é exibida
     */
    public function test_daily_data_section_displays()
    {
        $publicacao = Publicacao::factory()->create([
            'data_inicio' => Carbon::now()->subDays(10),
            'data_fim' => Carbon::now()->addDays(10),
        ]);

        // Criar dados dos últimos 7 dias
        for ($i = 0; $i < 7; $i++) {
            AuditoriaSplash::create([
                'hs_serial' => 'HS001',
                'cl_mac_address' => 'AA:BB:CC:DD:EE:' . str_pad($i, 2, '0', STR_PAD_LEFT),
                'pag_click' => 'http://example.com',
                'created_at' => now()->subDays($i),
            ]);
        }

        $response = $this->actingAs($this->user)
            ->get(route('publicacoes.estatisticas', $publicacao));

        $response->assertViewHas('stats', function($stats) {
            return isset($stats['impressoes_por_dia']);
        });
    }

    /**
     * Teste: Seção Usuários que Visualizaram é exibida
     */
    public function test_users_who_viewed_section_displays()
    {
        $publicacao = Publicacao::factory()->create([
            'data_inicio' => Carbon::now()->subDays(10),
            'data_fim' => Carbon::now()->addDays(10),
        ]);

        // Criar dados
        for ($i = 0; $i < 10; $i++) {
            AuditoriaSplash::create([
                'hs_serial' => 'HS001',
                'cl_mac_address' => 'AA:BB:CC:DD:EE:' . str_pad($i, 2, '0', STR_PAD_LEFT),
                'pag_click' => 'http://example.com/page-' . $i,
                'imagem' => 'criativo_' . $i . '.jpg',
                'dsc_identity' => 'Equipment-1',
                'created_at' => now(),
            ]);
        }

        $response = $this->actingAs($this->user)
            ->get(route('publicacoes.estatisticas', $publicacao));

        $response->assertViewHas('stats', function($stats) {
            return isset($stats['logs_acesso']) && count($stats['logs_acesso']) > 0;
        });
    }

    /**
     * Teste: Estatísticas gerais estão corretas
     */
    public function test_general_statistics_are_correct()
    {
        $publicacao = Publicacao::factory()->create([
            'data_inicio' => Carbon::now()->subDays(10),
            'data_fim' => Carbon::now()->addDays(10),
        ]);

        // Criar 15 registros
        for ($i = 0; $i < 15; $i++) {
            AuditoriaSplash::create([
                'hs_serial' => 'HS001',
                'cl_mac_address' => 'AA:BB:CC:DD:EE:' . str_pad($i, 2, '0', STR_PAD_LEFT),
                'pag_click' => 'http://example.com',
                'created_at' => now(),
            ]);
        }

        $response = $this->actingAs($this->user)
            ->get(route('publicacoes.estatisticas', $publicacao));

        $response->assertViewHas('stats', function($stats) {
            return $stats['total_impressoes'] == 15
                && $stats['total_cliques'] == 15
                && $stats['ctr'] == 100;
        });
    }

    /**
     * Teste: Página retorna 404 para publicação inexistente
     */
    public function test_estatisticas_returns_404_for_nonexistent_publicacao()
    {
        $response = $this->actingAs($this->user)
            ->get(route('publicacoes.estatisticas', 99999));

        $response->assertStatus(404);
    }

    /**
     * Teste: Usuário não autenticado é redirecionado
     */
    public function test_unauthenticated_user_is_redirected()
    {
        $publicacao = Publicacao::factory()->create();

        $response = $this->get(route('publicacoes.estatisticas', $publicacao));

        $response->assertRedirect('/login');
    }

    /**
     * Teste: Cliques por hora são calculados
     */
    public function test_clicks_per_hour_are_calculated()
    {
        $publicacao = Publicacao::factory()->create([
            'data_inicio' => Carbon::now()->subDays(10),
            'data_fim' => Carbon::now()->addDays(10),
        ]);

        // Criar registros em diferentes horas
        for ($hour = 0; $hour < 24; $hour += 3) {
            AuditoriaSplash::create([
                'hs_serial' => 'HS001',
                'cl_mac_address' => 'AA:BB:CC:DD:EE:' . str_pad($hour, 2, '0', STR_PAD_LEFT),
                'pag_click' => 'http://example.com',
                'created_at' => now()->setHour($hour),
            ]);
        }

        $response = $this->actingAs($this->user)
            ->get(route('publicacoes.estatisticas', $publicacao));

        $response->assertViewHas('stats', function($stats) {
            return isset($stats['cliques_por_hora']) && count($stats['cliques_por_hora']) > 0;
        });
    }
}
