<?php

namespace Tests\Unit\Publicacao;

use Tests\TestCase;
use App\Models\Publicacao;
use App\Models\Criativo;
use App\Models\Empresa;
use App\Models\User;
use App\Models\WalledGarden;
use App\Models\PublicacaoImpressao;
use App\Models\PublicacaoClique;
use Illuminate\Foundation\Testing\RefreshDatabase;

class PublicacaoModelTest extends TestCase
{
    use RefreshDatabase;

    protected $publicacao;
    protected $criativo;
    protected $franquia;
    protected $cliente;
    protected $user;

    protected function setUp(): void
    {
        parent::setUp();

        $this->franquia = Empresa::factory()->create(['tipo_empresa' => 'franquia']);
        $this->cliente = Empresa::factory()->create(['tipo_empresa' => 'cliente']);
        $this->criativo = Criativo::factory()->create(['status_aprovacao' => 'aprovado']);
        $this->user = User::factory()->create();

        $this->publicacao = Publicacao::factory()->create([
            'criativo_id' => $this->criativo->id,
            'franquia_id' => $this->franquia->id,
            'cliente_id' => $this->cliente->id,
            'publicada_por' => $this->user->id,
            'status' => 'ativa',
            'data_inicio' => now()->subDays(5),
            'data_fim' => now()->addDays(10),
        ]);
    }

    public function test_publicacao_pertence_a_um_criativo()
    {
        $this->assertInstanceOf(Criativo::class, $this->publicacao->criativo);
        $this->assertEquals($this->criativo->id, $this->publicacao->criativo->id);
    }

    public function test_publicacao_pertence_a_uma_franquia()
    {
        $this->assertInstanceOf(Empresa::class, $this->publicacao->franquia);
        $this->assertEquals('franquia', $this->publicacao->franquia->tipo_empresa);
    }

    public function test_scope_ativas_retorna_apenas_publicacoes_ativas()
    {
        Publicacao::factory()->create(['status' => 'pausada']);
        Publicacao::factory()->create(['status' => 'expirada']);
        
        $ativas = Publicacao::ativas()->get();
        
        $this->assertCount(1, $ativas);
        $this->assertEquals('ativa', $ativas->first()->status);
    }

    public function test_scope_vigentes_retorna_publicacoes_no_periodo_atual()
    {
        Publicacao::factory()->create([
            'status' => 'ativa',
            'data_inicio' => now()->subDays(5),
            'data_fim' => now()->addDays(5),
        ]);

        Publicacao::factory()->create([
            'status' => 'ativa',
            'data_inicio' => now()->addDays(10),
            'data_fim' => now()->addDays(20),
        ]);

        $vigentes = Publicacao::vigentes()->get();
        
        $this->assertGreaterThanOrEqual(2, $vigentes->count());
    }

    public function test_is_vigente_retorna_true_para_publicacao_no_periodo()
    {
        $publicacao = Publicacao::factory()->create([
            'status' => 'ativa',
            'data_inicio' => now()->subDays(5),
            'data_fim' => now()->addDays(5),
        ]);

        $this->assertTrue($publicacao->isVigente());
    }

    public function test_is_expirada_retorna_true_para_publicacao_expirada()
    {
        $publicacao = Publicacao::factory()->create([
            'data_inicio' => now()->subDays(20),
            'data_fim' => now()->subDays(10),
        ]);

        $this->assertTrue($publicacao->isExpirada());
    }

    public function test_dias_restantes_calcula_dias_corretamente()
    {
        $publicacao = Publicacao::factory()->create([
            'data_inicio' => now(),
            'data_fim' => now()->addDays(10),
        ]);

        $this->assertEquals(10, $publicacao->diasRestantes());
    }

    public function test_aplicavel_para_equipamento_retorna_true_para_tipo_todos()
    {
        $publicacao = Publicacao::factory()->create(['tipo_alcance' => 'todos']);

        $this->assertTrue($publicacao->aplicavelParaEquipamento('ABC123'));
    }

    public function test_incrementar_visualizacoes_incrementa_contador()
    {
        $visualizacoesAntes = $this->publicacao->total_visualizacoes ?? 0;
        
        $this->publicacao->incrementarVisualizacoes(5);
        
        $this->publicacao->refresh();
        $this->assertEquals($visualizacoesAntes + 5, $this->publicacao->total_visualizacoes);
    }

    public function test_data_inicio_e_cast_para_datetime()
    {
        $this->assertInstanceOf(\Carbon\Carbon::class, $this->publicacao->data_inicio);
    }

    public function test_habilitar_clique_e_cast_para_boolean()
    {
        $publicacao = Publicacao::factory()->create(['habilitar_clique' => 1]);
        $this->assertTrue($publicacao->habilitar_clique);
        $this->assertIsBool($publicacao->habilitar_clique);
    }

    public function test_publicacao_pode_ser_soft_deleted()
    {
        $id = $this->publicacao->id;
        
        $this->publicacao->delete();
        
        $this->assertNull(Publicacao::find($id));
        $this->assertNotNull(Publicacao::withTrashed()->find($id));
    }
}
