#!/usr/bin/env php
<?php

/**
 * Script de Teste - Sistema de Emails Automáticos
 * ================================================
 * Verifica a implementação completa do sistema de emails
 * para notificação de cadastros (aprovação/rejeição/pendente)
 */

echo "\n";
echo "╔══════════════════════════════════════════════════════════════╗\n";
echo "║   TESTE: Sistema de Emails Automáticos - FreeFi Admin       ║\n";
echo "╚══════════════════════════════════════════════════════════════╝\n";
echo "\n";

$baseDir = __DIR__;
$errors = [];
$warnings = [];
$success = [];

echo "📁 Diretório base: $baseDir\n\n";

// =========================================
// FASE 1: Verificar Mailables
// =========================================
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "FASE 1: Verificando Classes Mailable\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

$mailables = [
    'CadastroAprovado' => 'app/Mail/CadastroAprovado.php',
    'CadastroRejeitado' => 'app/Mail/CadastroRejeitado.php',
    'NovoCadastroPendente' => 'app/Mail/NovoCadastroPendente.php',
];

foreach ($mailables as $name => $path) {
    $fullPath = "$baseDir/$path";
    if (file_exists($fullPath)) {
        $content = file_get_contents($fullPath);
        
        // Verificar implements ShouldQueue
        if (strpos($content, 'implements ShouldQueue') !== false) {
            echo "  ✅ $name - Implementa ShouldQueue (email em fila)\n";
            $success[] = "$name implementa ShouldQueue";
        } else {
            echo "  ⚠️  $name - NÃO implementa ShouldQueue (email síncrono)\n";
            $warnings[] = "$name não usa fila (pode atrasar resposta)";
        }
        
        // Verificar construtor com User
        if (strpos($content, 'public User $user') !== false) {
            echo "  ✅ $name - Recebe User no construtor\n";
            $success[] = "$name recebe User";
        } else {
            echo "  ❌ $name - NÃO recebe User no construtor\n";
            $errors[] = "$name não recebe dados do usuário";
        }
        
        // Verificar envelope() personalizado
        if (preg_match('/subject: \'(.+?)\'/', $content, $matches)) {
            echo "  ✅ $name - Subject: \"{$matches[1]}\"\n";
            $success[] = "$name tem subject personalizado";
        }
        
    } else {
        echo "  ❌ $name - Arquivo NÃO encontrado!\n";
        $errors[] = "Mailable $name não existe";
    }
    echo "\n";
}

// =========================================
// FASE 2: Verificar Views de Email
// =========================================
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "FASE 2: Verificando Views Blade dos Emails\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

$views = [
    'cadastro-aprovado' => 'resources/views/emails/cadastro-aprovado.blade.php',
    'cadastro-rejeitado' => 'resources/views/emails/cadastro-rejeitado.blade.php',
    'novo-cadastro-pendente' => 'resources/views/emails/novo-cadastro-pendente.blade.php',
];

foreach ($views as $name => $path) {
    $fullPath = "$baseDir/$path";
    if (file_exists($fullPath)) {
        $content = file_get_contents($fullPath);
        $lines = count(file($fullPath));
        
        echo "  ✅ $name.blade.php - Encontrado ($lines linhas)\n";
        
        // Verificar se é HTML válido
        if (strpos($content, '<!DOCTYPE html>') !== false) {
            echo "     ✓ Estrutura HTML completa\n";
        }
        
        // Verificar variáveis Blade
        preg_match_all('/\{\{\s*\$(\w+)\s*\}\}/', $content, $matches);
        if (!empty($matches[1])) {
            $vars = array_unique($matches[1]);
            echo "     ✓ Variáveis Blade: " . implode(', ', $vars) . "\n";
            $success[] = "View $name usa " . count($vars) . " variáveis";
        }
        
        // Verificar CSS inline
        if (strpos($content, '<style>') !== false) {
            echo "     ✓ CSS inline incluído (compatibilidade com clientes email)\n";
        }
        
    } else {
        echo "  ❌ $name.blade.php - NÃO encontrado!\n";
        $errors[] = "View $name não existe";
    }
    echo "\n";
}

// =========================================
// FASE 3: Verificar Integração no UserController
// =========================================
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "FASE 3: Verificando Integração no UserController\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

$userControllerPath = "$baseDir/app/Http/Controllers/UserController.php";
if (file_exists($userControllerPath)) {
    $content = file_get_contents($userControllerPath);
    
    // Verificar imports
    if (strpos($content, 'use App\Mail\CadastroAprovado') !== false) {
        echo "  ✅ Import: CadastroAprovado\n";
        $success[] = "UserController importa CadastroAprovado";
    } else {
        echo "  ❌ Falta import: CadastroAprovado\n";
        $errors[] = "UserController não importa CadastroAprovado";
    }
    
    if (strpos($content, 'use App\Mail\CadastroRejeitado') !== false) {
        echo "  ✅ Import: CadastroRejeitado\n";
        $success[] = "UserController importa CadastroRejeitado";
    } else {
        echo "  ❌ Falta import: CadastroRejeitado\n";
        $errors[] = "UserController não importa CadastroRejeitado";
    }
    
    if (strpos($content, 'use Illuminate\Support\Facades\Mail') !== false) {
        echo "  ✅ Import: Mail Facade\n";
        $success[] = "UserController importa Mail";
    } else {
        echo "  ❌ Falta import: Mail Facade\n";
        $errors[] = "UserController não importa Mail";
    }
    
    echo "\n";
    
    // Verificar método approve()
    if (preg_match('/public function approve\(User \$user\)/', $content)) {
        echo "  ✅ Método approve() encontrado\n";
        
        if (strpos($content, 'Mail::to($user->email)->queue(new CadastroAprovado($user))') !== false) {
            echo "     ✓ Email de aprovação sendo enviado (com queue)\n";
            $success[] = "approve() envia email em fila";
        } elseif (strpos($content, 'Mail::to') !== false && strpos($content, 'CadastroAprovado') !== false) {
            echo "     ⚠️  Email de aprovação implementado mas não usa queue\n";
            $warnings[] = "approve() não usa fila para email";
        } else {
            echo "     ❌ Email de aprovação NÃO está sendo enviado\n";
            $errors[] = "approve() não envia email";
        }
    }
    
    echo "\n";
    
    // Verificar método reject()
    if (preg_match('/public function reject\(User \$user/', $content)) {
        echo "  ✅ Método reject() encontrado\n";
        
        if (strpos($content, 'Mail::to($user->email)->queue(new CadastroRejeitado') !== false) {
            echo "     ✓ Email de rejeição sendo enviado (com queue)\n";
            $success[] = "reject() envia email em fila";
        } elseif (strpos($content, 'Mail::to') !== false && strpos($content, 'CadastroRejeitado') !== false) {
            echo "     ⚠️  Email de rejeição implementado mas não usa queue\n";
            $warnings[] = "reject() não usa fila para email";
        } else {
            echo "     ❌ Email de rejeição NÃO está sendo enviado\n";
            $errors[] = "reject() não envia email";
        }
        
        // Verificar se aceita parâmetro motivo
        if (strpos($content, 'Request $request') !== false && 
            strpos($content, '$motivo') !== false) {
            echo "     ✓ Aceita parâmetro 'motivo' opcional\n";
            $success[] = "reject() aceita motivo da rejeição";
        }
    }
    
} else {
    echo "  ❌ UserController.php NÃO encontrado!\n";
    $errors[] = "UserController não existe";
}

echo "\n";

// =========================================
// FASE 4: Verificar Integração no RegisterController
// =========================================
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "FASE 4: Verificando Integração no RegisterController\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

$registerControllerPath = "$baseDir/app/Http/Controllers/Auth/RegisterController.php";
if (file_exists($registerControllerPath)) {
    $content = file_get_contents($registerControllerPath);
    
    // Verificar import
    if (strpos($content, 'use App\Mail\NovoCadastroPendente') !== false) {
        echo "  ✅ Import: NovoCadastroPendente\n";
        $success[] = "RegisterController importa NovoCadastroPendente";
    } else {
        echo "  ❌ Falta import: NovoCadastroPendente\n";
        $errors[] = "RegisterController não importa NovoCadastroPendente";
    }
    
    if (strpos($content, 'use Illuminate\Support\Facades\Mail') !== false) {
        echo "  ✅ Import: Mail Facade\n";
        $success[] = "RegisterController importa Mail";
    } else {
        echo "  ❌ Falta import: Mail Facade\n";
        $errors[] = "RegisterController não importa Mail";
    }
    
    echo "\n";
    
    // Verificar método register()
    if (preg_match('/public function register\(/', $content)) {
        echo "  ✅ Método register() encontrado\n";
        
        // Verificar busca de admins
        if (strpos($content, "User::where('is_superadmin', true)") !== false ||
            strpos($content, '$admins = User::') !== false) {
            echo "     ✓ Busca administradores para notificar\n";
            $success[] = "register() busca admins";
        } else {
            echo "     ⚠️  Não busca administradores\n";
            $warnings[] = "register() não busca admins para notificar";
        }
        
        // Verificar envio de email
        if (strpos($content, 'Mail::to($admin->email)->queue(new NovoCadastroPendente') !== false) {
            echo "     ✓ Notifica admins sobre novo cadastro (com queue)\n";
            $success[] = "register() notifica admins em fila";
        } elseif (strpos($content, 'NovoCadastroPendente') !== false) {
            echo "     ⚠️  Notifica admins mas não usa queue\n";
            $warnings[] = "register() não usa fila para notificação";
        } else {
            echo "     ❌ NÃO notifica admins sobre novo cadastro\n";
            $errors[] = "register() não notifica admins";
        }
        
        // Verificar loop foreach
        if (strpos($content, 'foreach ($admins as $admin)') !== false) {
            echo "     ✓ Loop foreach para enviar email a todos admins\n";
            $success[] = "register() envia para todos admins";
        }
    }
    
} else {
    echo "  ❌ RegisterController.php NÃO encontrado!\n";
    $errors[] = "RegisterController não existe";
}

echo "\n";

// =========================================
// FASE 5: Verificar Configuração de Email
// =========================================
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "FASE 5: Verificando Configuração de Email (.env)\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

$envPath = "$baseDir/.env";
if (file_exists($envPath)) {
    $envContent = file_get_contents($envPath);
    
    $mailConfigs = [
        'MAIL_MAILER' => false,
        'MAIL_HOST' => false,
        'MAIL_PORT' => false,
        'MAIL_USERNAME' => false,
        'MAIL_PASSWORD' => false,
        'MAIL_ENCRYPTION' => false,
        'MAIL_FROM_ADDRESS' => false,
    ];
    
    foreach ($mailConfigs as $key => $found) {
        if (preg_match("/$key=(.+)/", $envContent, $matches)) {
            $value = trim($matches[1]);
            if (!empty($value) && $value !== 'null') {
                echo "  ✅ $key=$value\n";
                $mailConfigs[$key] = true;
            } else {
                echo "  ⚠️  $key está vazio ou null\n";
                $warnings[] = "$key não configurado no .env";
            }
        } else {
            echo "  ⚠️  $key não encontrado no .env\n";
            $warnings[] = "$key ausente no .env";
        }
    }
    
    $configured = array_filter($mailConfigs);
    $totalConfigs = count($mailConfigs);
    $configuredCount = count($configured);
    
    echo "\n";
    if ($configuredCount === $totalConfigs) {
        echo "  ✅ Todas as configurações de email estão presentes!\n";
        $success[] = "Configuração de email completa";
    } elseif ($configuredCount > 0) {
        echo "  ⚠️  Configuração de email parcial ($configuredCount/$totalConfigs)\n";
        $warnings[] = "Configuração de email incompleta";
    } else {
        echo "  ❌ Nenhuma configuração de email encontrada!\n";
        $errors[] = "Email não configurado no .env";
    }
    
} else {
    echo "  ⚠️  Arquivo .env não encontrado!\n";
    $warnings[] = "Arquivo .env não existe";
}

echo "\n";

// =========================================
// FASE 6: Verificar Queue Configuration
// =========================================
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "FASE 6: Verificando Configuração de Filas\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

if (file_exists($envPath)) {
    $envContent = file_get_contents($envPath);
    
    if (preg_match('/QUEUE_CONNECTION=(.+)/', $envContent, $matches)) {
        $queueDriver = trim($matches[1]);
        echo "  ✅ QUEUE_CONNECTION=$queueDriver\n";
        
        if ($queueDriver === 'sync') {
            echo "     ⚠️  Usando 'sync' - emails serão enviados imediatamente (não usa fila real)\n";
            $warnings[] = "Queue em modo 'sync' (não é assíncrono de verdade)";
        } elseif ($queueDriver === 'database') {
            echo "     ✓ Usando 'database' - emails serão enfileirados no banco\n";
            echo "     📌 Lembre-se de rodar: php artisan queue:work\n";
            $success[] = "Queue configurada para database";
        } elseif ($queueDriver === 'redis') {
            echo "     ✓ Usando 'redis' - emails serão enfileirados no Redis\n";
            echo "     📌 Lembre-se de rodar: php artisan queue:work\n";
            $success[] = "Queue configurada para Redis";
        }
    } else {
        echo "  ⚠️  QUEUE_CONNECTION não encontrado (usando padrão 'sync')\n";
        $warnings[] = "QUEUE_CONNECTION não definido";
    }
}

echo "\n";

// =========================================
// RELATÓRIO FINAL
// =========================================
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "RELATÓRIO FINAL\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

$totalChecks = count($success) + count($warnings) + count($errors);
$successRate = $totalChecks > 0 ? round((count($success) / $totalChecks) * 100, 1) : 0;

echo "📊 ESTATÍSTICAS:\n";
echo "   ✅ Sucessos: " . count($success) . "\n";
echo "   ⚠️  Avisos: " . count($warnings) . "\n";
echo "   ❌ Erros: " . count($errors) . "\n";
echo "   📈 Taxa de sucesso: {$successRate}%\n";
echo "\n";

if (!empty($errors)) {
    echo "❌ ERROS CRÍTICOS:\n";
    foreach ($errors as $i => $error) {
        echo "   " . ($i + 1) . ". $error\n";
    }
    echo "\n";
}

if (!empty($warnings)) {
    echo "⚠️  AVISOS:\n";
    foreach ($warnings as $i => $warning) {
        echo "   " . ($i + 1) . ". $warning\n";
    }
    echo "\n";
}

// Status final
if (count($errors) === 0) {
    if (count($warnings) === 0) {
        echo "✅ PERFEITO! Sistema de emails completamente implementado!\n";
        echo "   Todos os componentes estão presentes e configurados.\n";
        exit(0);
    } else {
        echo "✅ BOM! Sistema de emails implementado com avisos.\n";
        echo "   Funcionalidade básica está presente, mas há melhorias possíveis.\n";
        exit(0);
    }
} else {
    echo "❌ ATENÇÃO! Há erros críticos que precisam ser corrigidos.\n";
    echo "   O sistema de emails pode não funcionar corretamente.\n";
    exit(1);
}
