<?php

require __DIR__.'/vendor/autoload.php';

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;

$app = require_once __DIR__.'/bootstrap/app.php';
$app->make(\Illuminate\Contracts\Console\Kernel::class)->bootstrap();

echo "\n";
echo "╔══════════════════════════════════════════════════════════════════╗\n";
echo "║         TESTE - MIDDLEWARE DE BLOQUEIO POR APROVAÇÃO            ║\n";
echo "╚══════════════════════════════════════════════════════════════════╝\n";
echo "\n";

try {
    echo "FASE 1: Verificar Middleware Criado\n";
    echo "══════════════════════════════════════════════════════════════════════\n";
    
    $middlewarePath = app_path('Http/Middleware/CheckApprovalStatus.php');
    if (file_exists($middlewarePath)) {
        echo "✓ Middleware criado: CheckApprovalStatus.php\n";
        
        $content = file_get_contents($middlewarePath);
        
        if (strpos($content, 'status_cadastro') !== false) {
            echo "✓ Verifica campo 'status_cadastro'\n";
        }
        
        if (strpos($content, "!== 'aprovado'") !== false) {
            echo "✓ Bloqueia usuários não aprovados\n";
        }
        
        if (strpos($content, 'Auth::logout()') !== false) {
            echo "✓ Faz logout automático\n";
        }
        
        if (strpos($content, 'pendente') !== false && strpos($content, 'rejeitado') !== false) {
            echo "✓ Mensagens personalizadas para pendente/rejeitado\n";
        }
        
        if (strpos($content, "redirect()->route('login')") !== false) {
            echo "✓ Redireciona para tela de login\n";
        }
    } else {
        echo "✗ ERRO: Middleware não encontrado\n";
    }
    
    echo "\nFASE 2: Verificar Registro no Bootstrap\n";
    echo "══════════════════════════════════════════════════════════════════════\n";
    
    $bootstrapPath = base_path('bootstrap/app.php');
    $bootstrapContent = file_get_contents($bootstrapPath);
    
    if (strpos($bootstrapContent, 'check.approval') !== false) {
        echo "✓ Alias 'check.approval' registrado\n";
    } else {
        echo "✗ ERRO: Alias não encontrado\n";
    }
    
    if (strpos($bootstrapContent, 'CheckApprovalStatus') !== false) {
        echo "✓ Middleware CheckApprovalStatus vinculado\n";
    } else {
        echo "✗ ERRO: Middleware não vinculado\n";
    }
    
    echo "\nFASE 3: Verificar Rotas Protegidas\n";
    echo "══════════════════════════════════════════════════════════════════════\n";
    
    $routesPath = base_path('routes/web.php');
    $routesContent = file_get_contents($routesPath);
    
    if (strpos($routesContent, "'check.approval'") !== false) {
        echo "✓ Middleware aplicado nas rotas\n";
        
        // Contar ocorrências
        $count = substr_count($routesContent, 'check.approval');
        echo "ℹ   → Aplicado em {$count} local(is)\n";
    } else {
        echo "✗ ERRO: Middleware não aplicado nas rotas\n";
    }
    
    echo "\nFASE 4: Estatísticas de Usuários por Status\n";
    echo "══════════════════════════════════════════════════════════════════════\n";
    
    $stats = [
        'aprovado' => DB::table('users')->where('status_cadastro', 'aprovado')->count(),
        'pendente' => DB::table('users')->where('status_cadastro', 'pendente')->count(),
        'rejeitado' => DB::table('users')->where('status_cadastro', 'rejeitado')->count(),
    ];
    
    echo "✓ Estatísticas coletadas:\n";
    echo "ℹ   → 🟢 Aprovados: {$stats['aprovado']} (PODEM logar)\n";
    echo "ℹ   → 🟡 Pendentes: {$stats['pendente']} (BLOQUEADOS pelo middleware)\n";
    echo "ℹ   → 🔴 Rejeitados: {$stats['rejeitado']} (BLOQUEADOS pelo middleware)\n";
    
    echo "\nFASE 5: Listar Usuários Bloqueados\n";
    echo "══════════════════════════════════════════════════════════════════════\n";
    
    $bloqueados = DB::table('users')
        ->whereIn('status_cadastro', ['pendente', 'rejeitado'])
        ->get(['id', 'name', 'email', 'status_cadastro']);
    
    if ($bloqueados->count() > 0) {
        echo "✓ Usuários que NÃO podem logar:\n\n";
        
        foreach ($bloqueados as $user) {
            $status = $user->status_cadastro === 'pendente' ? '🟡 PENDENTE' : '🔴 REJEITADO';
            echo "  [$status] {$user->name} ({$user->email})\n";
            
            if ($user->status_cadastro === 'pendente') {
                echo "    → Mensagem: 'Seu cadastro está aguardando aprovação'\n";
            } else {
                echo "    → Mensagem: 'Seu cadastro foi rejeitado'\n";
            }
        }
    } else {
        echo "ℹ Todos os usuários estão aprovados (nenhum bloqueio)\n";
    }
    
    echo "\nFASE 6: Listar Usuários Aprovados (Podem Logar)\n";
    echo "══════════════════════════════════════════════════════════════════════\n";
    
    $aprovados = DB::table('users')
        ->where('status_cadastro', 'aprovado')
        ->get(['id', 'name', 'email']);
    
    if ($aprovados->count() > 0) {
        echo "✓ Usuários que PODEM logar:\n\n";
        
        foreach ($aprovados as $user) {
            echo "  [🟢 APROVADO] {$user->name} ({$user->email})\n";
        }
    }
    
    echo "\n";
    echo "╔══════════════════════════════════════════════════════════════════╗\n";
    echo "║                     RESUMO DO MIDDLEWARE                         ║\n";
    echo "╚══════════════════════════════════════════════════════════════════╝\n";
    echo "\n";
    
    echo "FUNCIONAMENTO:\n";
    echo "══════════════════════════════════════════════════════════════════════\n";
    echo "  1. Usuário faz login com email/senha\n";
    echo "  2. Laravel autentica credenciais ✓\n";
    echo "  3. Middleware 'check.approval' é executado\n";
    echo "  4. Verifica: status_cadastro === 'aprovado'?\n";
    echo "     ├─ SIM (aprovado) → Permite acesso ✅\n";
    echo "     └─ NÃO (pendente/rejeitado) → Bloqueia ❌\n";
    echo "  5. Se bloqueado:\n";
    echo "     ├─ Auth::logout() (desconecta)\n";
    echo "     ├─ Invalida sessão\n";
    echo "     ├─ Regenera token CSRF\n";
    echo "     └─ Redireciona para /login com mensagem de erro\n";
    echo "\n";
    
    echo "MENSAGENS DE ERRO:\n";
    echo "══════════════════════════════════════════════════════════════════════\n";
    echo "  🟡 Pendente:\n";
    echo "     'Seu cadastro está aguardando aprovação.\n";
    echo "      Você receberá um email quando for aprovado.'\n";
    echo "\n";
    echo "  🔴 Rejeitado:\n";
    echo "     'Seu cadastro foi rejeitado.\n";
    echo "      Entre em contato com o administrador.'\n";
    echo "\n";
    
    echo "ROTAS PROTEGIDAS:\n";
    echo "══════════════════════════════════════════════════════════════════════\n";
    echo "  ✓ /dashboard\n";
    echo "  ✓ /profile\n";
    echo "  ✓ /users/*\n";
    echo "  ✓ /empresas/*\n";
    echo "  ✓ /midias/*\n";
    echo "  ✓ /hotspots/*\n";
    echo "  ✓ /groups/*\n";
    echo "  ✓ /permissions/*\n";
    echo "\n";
    
    echo "TESTE MANUAL:\n";
    echo "══════════════════════════════════════════════════════════════════════\n";
    
    $usuarioPendente = DB::table('users')->where('status_cadastro', 'pendente')->first();
    
    if ($usuarioPendente) {
        echo "  1. Tente fazer login com usuário PENDENTE:\n";
        echo "     Email: {$usuarioPendente->email}\n";
        echo "     Senha: password\n";
        echo "\n";
        echo "  2. Resultado esperado:\n";
        echo "     ❌ Login bloqueado\n";
        echo "     🔴 Mensagem de erro na tela de login\n";
        echo "     ↩️  Redirecionado de volta para /login\n";
    } else {
        echo "  ℹ Nenhum usuário pendente para teste\n";
        echo "  💡 Crie um novo usuário em /register para testar\n";
    }
    
    echo "\n";
    echo "PRÓXIMOS PASSOS:\n";
    echo "══════════════════════════════════════════════════════════════════════\n";
    echo "  📧 Implementar emails de aprovação/rejeição\n";
    echo "  📊 Dashboard com estatísticas de aprovações\n";
    echo "  🔔 Notificações in-app para admins\n";
    echo "  📜 Log de histórico de aprovações\n";
    echo "\n";
    
} catch (\Exception $e) {
    echo "\n❌ ERRO: " . $e->getMessage() . "\n";
    echo "Stack trace:\n" . $e->getTraceAsString() . "\n";
}
