#!/usr/bin/env php
<?php

/**
 * Teste de Conexão SMTP
 * =====================
 * Verifica se o servidor SMTP está configurado corretamente
 */

require __DIR__.'/vendor/autoload.php';

$app = require_once __DIR__.'/bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Config;

echo "\n";
echo "╔═══════════════════════════════════════════════════════════╗\n";
echo "║          TESTE DE CONEXÃO SMTP - FreeFi Admin            ║\n";
echo "╚═══════════════════════════════════════════════════════════╝\n";
echo "\n";

// Mostrar configurações atuais
echo "📧 CONFIGURAÇÕES DE EMAIL:\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "  MAIL_MAILER:       " . Config::get('mail.default') . "\n";
echo "  MAIL_HOST:         " . Config::get('mail.mailers.smtp.host') . "\n";
echo "  MAIL_PORT:         " . Config::get('mail.mailers.smtp.port') . "\n";
echo "  MAIL_USERNAME:     " . Config::get('mail.mailers.smtp.username') . "\n";
echo "  MAIL_ENCRYPTION:   " . Config::get('mail.mailers.smtp.encryption') . "\n";
echo "  MAIL_FROM_ADDRESS: " . Config::get('mail.from.address') . "\n";
echo "  MAIL_FROM_NAME:    " . Config::get('mail.from.name') . "\n";
echo "\n";

// Verificar configurações
$errors = [];
$warnings = [];

if (Config::get('mail.default') !== 'smtp') {
    $errors[] = "MAIL_MAILER não está configurado como 'smtp'";
}

if (empty(Config::get('mail.mailers.smtp.host'))) {
    $errors[] = "MAIL_HOST está vazio";
}

if (empty(Config::get('mail.mailers.smtp.username'))) {
    $warnings[] = "MAIL_USERNAME está vazio (pode ser necessário para SMTP autenticado)";
}

if (empty(Config::get('mail.mailers.smtp.password'))) {
    $warnings[] = "MAIL_PASSWORD está vazio (pode ser necessário para SMTP autenticado)";
}

if (!empty($errors)) {
    echo "❌ ERROS CRÍTICOS:\n";
    foreach ($errors as $error) {
        echo "   • $error\n";
    }
    echo "\n";
    exit(1);
}

if (!empty($warnings)) {
    echo "⚠️  AVISOS:\n";
    foreach ($warnings as $warning) {
        echo "   • $warning\n";
    }
    echo "\n";
}

// Perguntar qual email enviar
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
echo "🧪 TESTE DE ENVIO\n";
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";

echo "Digite o email de destino para teste: ";
$emailDestino = trim(fgets(STDIN));

if (empty($emailDestino) || !filter_var($emailDestino, FILTER_VALIDATE_EMAIL)) {
    echo "❌ Email inválido!\n";
    exit(1);
}

echo "\n";
echo "📤 Enviando email de teste para: $emailDestino\n";
echo "   Aguarde...\n\n";

try {
    // Tentar enviar email de teste
    Mail::raw('Este é um email de teste do sistema FreeFi Admin.', function($message) use ($emailDestino) {
        $message->to($emailDestino)
                ->subject('🧪 Teste de Conexão SMTP - FreeFi Admin')
                ->from(
                    Config::get('mail.from.address'),
                    Config::get('mail.from.name')
                );
    });
    
    echo "✅ EMAIL ENVIADO COM SUCESSO!\n\n";
    echo "📋 Próximos passos:\n";
    echo "   1. Verifique a caixa de entrada de: $emailDestino\n";
    echo "   2. Verifique também a pasta de SPAM\n";
    echo "   3. O assunto do email é: '🧪 Teste de Conexão SMTP - FreeFi Admin'\n";
    echo "\n";
    echo "✅ Conexão SMTP está funcionando corretamente!\n";
    echo "   O sistema está pronto para enviar emails automáticos.\n\n";
    
    exit(0);
    
} catch (\Exception $e) {
    echo "❌ ERRO AO ENVIAR EMAIL!\n\n";
    echo "📋 Detalhes do erro:\n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    echo $e->getMessage() . "\n";
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
    
    echo "🔧 POSSÍVEIS SOLUÇÕES:\n";
    echo "   1. Verificar se o servidor SMTP está acessível\n";
    echo "   2. Verificar usuário e senha no .env\n";
    echo "   3. Verificar se a porta está correta (465 para SSL, 587 para TLS)\n";
    echo "   4. Verificar firewall/iptables\n";
    echo "   5. Verificar logs do Laravel: storage/logs/laravel.log\n\n";
    
    // Testar conexão básica
    echo "🔍 Testando conexão TCP com o servidor SMTP...\n";
    $host = Config::get('mail.mailers.smtp.host');
    $port = Config::get('mail.mailers.smtp.port');
    
    $connection = @fsockopen($host, $port, $errno, $errstr, 10);
    
    if ($connection) {
        echo "   ✅ Conexão TCP com $host:$port estabelecida!\n";
        echo "   O problema pode estar nas credenciais (usuário/senha).\n";
        fclose($connection);
    } else {
        echo "   ❌ Falha na conexão TCP com $host:$port\n";
        echo "   Erro: [$errno] $errstr\n";
        echo "   O servidor SMTP pode estar bloqueado ou inacessível.\n";
    }
    
    echo "\n";
    exit(1);
}
