<?php

/**
 * Teste Rápido - Nova View de Usuários
 * Verifica se o layout atualizado está funcionando
 */

require __DIR__.'/vendor/autoload.php';

$app = require_once __DIR__.'/bootstrap/app.php';
$kernel = $app->make(\Illuminate\Contracts\Http\Kernel::class);
$kernel->bootstrap();

use Illuminate\Http\Request;
use App\Models\User;

class TestOutput {
    public static function success($msg) { echo "\033[32m✓\033[0m {$msg}\n"; }
    public static function error($msg) { echo "\033[31m✗\033[0m {$msg}\n"; }
    public static function info($msg) { echo "\033[36mℹ\033[0m {$msg}\n"; }
    public static function title($msg) { echo "\n\033[1;33m{$msg}\033[0m\n" . str_repeat('═', 70) . "\n"; }
}

$passed = 0;
$failed = 0;

echo "\033[1m\n╔══════════════════════════════════════════════════════════════════╗\n";
echo "║         TESTE - NOVA VIEW DE USUÁRIOS (Layout Melhorado)        ║\n";
echo "╚══════════════════════════════════════════════════════════════════╝\033[0m\n";

TestOutput::title('TESTE 1: Verificar GET /users');

// Autenticar
$user = User::where('status_cadastro', 'aprovado')->first();
auth()->login($user);

$request = Request::create('/users', 'GET');
$response = $kernel->handle($request);

if ($response->getStatusCode() === 200) {
    TestOutput::success("GET /users retornou status 200");
    $passed++;
    
    $content = $response->getContent();
    
    // Verificar elementos do novo layout
    $checks = [
        'Gestão de Usuários' => 'Breadcrumb presente',
        'Usuários Cadastrados' => 'Título do card presente',
        'Buscar por nome ou email' => 'Campo de busca presente',
        'Novo Usuário' => 'Botão de criar presente',
        'ph-trash' => 'Botão de lixeira presente',
        'badge bg-' => 'Badges de status presentes',
        'ph-pencil-simple' => 'Ícone de editar presente',
        'trashModalUsers' => 'Modal de lixeira presente',
    ];
    
    foreach ($checks as $needle => $description) {
        if (strpos($content, $needle) !== false) {
            TestOutput::success($description);
            $passed++;
        } else {
            TestOutput::error($description . " - NÃO ENCONTRADO!");
            $failed++;
        }
    }
    
    // Verificar se NÃO tem erros
    if (strpos($content, 'currentPage does not exist') === false) {
        TestOutput::success("Sem erro de Collection/Paginator");
        $passed++;
    } else {
        TestOutput::error("Erro de paginação detectado!");
        $failed++;
    }
    
} else {
    TestOutput::error("GET /users retornou status " . $response->getStatusCode());
    $failed++;
}

TestOutput::title('TESTE 2: Verificar Rota de Busca');

$request = Request::create('/users?q=test', 'GET');
$response = $kernel->handle($request);

if ($response->getStatusCode() === 200) {
    TestOutput::success("GET /users?q=test retornou status 200");
    $passed++;
} else {
    TestOutput::error("Busca falhou com status " . $response->getStatusCode());
    $failed++;
}

TestOutput::title('TESTE 3: Verificar Rota de Lixeira (JSON)');

$request = Request::create('/users/trash', 'GET');
$response = $kernel->handle($request);

if ($response->getStatusCode() === 200) {
    TestOutput::success("GET /users/trash retornou status 200");
    $passed++;
    
    $json = json_decode($response->getContent(), true);
    
    if (is_array($json)) {
        TestOutput::success("Retornou JSON válido");
        TestOutput::info("  → Usuários na lixeira: " . count($json));
        $passed++;
    } else {
        TestOutput::error("JSON inválido");
        $failed++;
    }
} else {
    TestOutput::error("GET /users/trash falhou com status " . $response->getStatusCode());
    $failed++;
}

TestOutput::title('TESTE 4: Verificar Estrutura da View');

$viewPath = resource_path('views/users/index.blade.php');

if (file_exists($viewPath)) {
    TestOutput::success("View users/index.blade.php existe");
    $passed++;
    
    $viewContent = file_get_contents($viewPath);
    
    // Verificar estrutura esperada
    $estrutura = [
        '@extends(\'layouts.master\')' => 'Extends layout master',
        '@section(\'title\', \'Usuários\')' => 'Define título da página',
        'app-line-breadcrumbs' => 'Breadcrumb presente',
        'table-responsive' => 'Tabela responsiva',
        'card-footer' => 'Footer com paginação',
        '@push(\'scripts\')' => 'Scripts JavaScript incluídos',
        'trashModalUsers' => 'Modal de lixeira configurado',
        'delete-user' => 'Classe de exclusão',
        'restore-user' => 'Classe de restauração',
    ];
    
    foreach ($estrutura as $needle => $description) {
        if (strpos($viewContent, $needle) !== false) {
            TestOutput::success($description);
            $passed++;
        } else {
            TestOutput::error($description . " - NÃO ENCONTRADO!");
            $failed++;
        }
    }
    
} else {
    TestOutput::error("View users/index.blade.php não existe!");
    $failed++;
}

// Desautenticar
auth()->logout();

echo "\n\033[1m╔══════════════════════════════════════════════════════════════════╗\n";
echo "║                        RESUMO DO TESTE                           ║\n";
echo "╚══════════════════════════════════════════════════════════════════╝\033[0m\n\n";

echo "Total de testes: \033[1m" . ($passed + $failed) . "\033[0m\n";
echo "\033[32m✓ Passados: {$passed}\033[0m\n";

if ($failed > 0) {
    echo "\033[31m✗ Falhados: {$failed}\033[0m\n";
}

$percentage = ($passed + $failed) > 0 ? round(($passed / ($passed + $failed)) * 100, 2) : 0;

echo "\n\033[1mTaxa de sucesso: ";
if ($percentage === 100.0) {
    echo "\033[32m{$percentage}%\033[0m ✓\n\n";
} else {
    echo "\033[33m{$percentage}%\033[0m ⚠️\n\n";
}

echo "\033[36m══════════════════════════════════════════════════════════════════\033[0m\n";
echo "\033[1mMELHORIAS IMPLEMENTADAS:\033[0m\n";
echo "  ✓ Breadcrumb com ícone de usuários\n";
echo "  ✓ Campo de busca por nome ou email\n";
echo "  ✓ Botão de lixeira (modal)\n";
echo "  ✓ Coluna de empresa associada\n";
echo "  ✓ Badges de status coloridos (pendente/aprovado/rejeitado)\n";
echo "  ✓ Ícones modernos (Phosphor Icons)\n";
echo "  ✓ Paginação com links\n";
echo "  ✓ Modal de lixeira com restauração\n";
echo "  ✓ Layout consistente com empresas e mídias\n";
echo "\033[36m══════════════════════════════════════════════════════════════════\033[0m\n\n";

exit($failed > 0 ? 1 : 0);
