# PLANO DE DEPLOY - SERVIDOR LINODE/AKAMAI
## FreeFi Admin - Ambiente de Produção

---

**Data:** 21/10/2025
**Servidor:** Linode Cloud (Akamai)
**Ambiente:** Produção (sem homologação - deploy direto)
**Status:** 🔴 AGUARDANDO CREDENCIAIS

---

## ⚠️ IMPORTANTE - LEIA ANTES DE COMEÇAR

**Este deploy será feito DIRETAMENTE em produção sem ambiente de homologação.**

Antes de iniciar o deploy, é **OBRIGATÓRIO** completar as **correções de segurança críticas** documentadas no arquivo `analise-cloudeia.txt`, especialmente:

1. ✅ Rotacionar todas as credenciais expostas
2. ✅ Configurar APP_DEBUG=false para produção
3. ✅ Implementar policies de autorização
4. ✅ Adicionar rate limiting nas APIs
5. ✅ Remover .env do repositório Git

**Não prossiga sem completar as correções críticas!**

---

## 📋 CREDENCIAIS NECESSÁRIAS

Para executar este plano, precisaremos das seguintes informações:

### 1. Acesso SSH ao Servidor Linode
```
SSH_HOST=_____________________________ (IP ou hostname)
SSH_PORT=_____________________________ (padrão: 22)
SSH_USERNAME=_________________________ (usuário SSH)
SSH_PRIVATE_KEY=______________________ (chave privada ou senha)
```

### 2. Informações do Servidor
```
SISTEMA_OPERACIONAL=__________________ (Ubuntu, Debian, CentOS, etc)
VERSAO_OS=____________________________ (22.04, 24.04, etc)
RAM_DISPONIVEL=_______________________ (GB)
CPU_CORES=____________________________ (número de cores)
DISK_SPACE=___________________________ (GB disponíveis)
```

### 3. Domínio e DNS
```
DOMAIN_NAME=__________________________ (ex: admin.freefi.com.br)
DNS_CONFIGURADO=______________________ (Sim/Não)
CLOUDFLARE_ENABLED=___________________ (Sim/Não)
```

### 4. Banco de Dados
```
Opção escolhida:
[ ] Usar MySQL no Docker (recomendado para começar)
[ ] Usar MySQL gerenciado (Linode Database, AWS RDS, etc)

Se usar MySQL gerenciado:
DB_HOST=______________________________
DB_PORT=______________________________
DB_DATABASE=__________________________
DB_USERNAME=__________________________
DB_PASSWORD=__________________________
```

### 5. Email (SMTP)
```
MAIL_HOST=____________________________
MAIL_PORT=____________________________
MAIL_USERNAME=________________________
MAIL_PASSWORD=________________________
MAIL_FROM_ADDRESS=____________________
MAIL_ENCRYPTION=_______________________ (tls/ssl)
```

### 6. GitHub (para CI/CD)
```
GITHUB_USERNAME=______________________
GITHUB_REPO=__________________________
GITHUB_PAT=___________________________ (Personal Access Token)
```

### 7. Secrets para GitHub Actions
Estes secrets serão configurados no repositório GitHub:
```
SSH_HOST=_____________________________ (mesmo do item 1)
SSH_USERNAME=_________________________ (mesmo do item 1)
SSH_PRIVATE_KEY=______________________ (mesmo do item 1)
SSH_PORT=_____________________________ (mesmo do item 1)
PROJECT_PATH=_________________________ (caminho no servidor, ex: /var/www/freefi)
GH_PAT=_______________________________ (GitHub Personal Access Token)
```

---

## 📊 PRÉ-REQUISITOS DO SERVIDOR

### Requisitos Mínimos Recomendados:
- **CPU:** 2 vCPUs (mínimo) / 4 vCPUs (recomendado)
- **RAM:** 4 GB (mínimo) / 8 GB (recomendado)
- **Disco:** 50 GB SSD (mínimo) / 100 GB (recomendado)
- **OS:** Ubuntu 22.04 LTS ou 24.04 LTS
- **Firewall:** Portas 80, 443, 22 abertas

### Software Necessário:
- Docker Engine (latest)
- Docker Compose V2
- Git
- UFW (Uncomplicated Firewall)
- Certbot (para SSL/Let's Encrypt)
- Nginx (ou será usado o container Nginx)

---

## 🚀 PLANO DE DEPLOY - ETAPAS

### FASE 0: PREPARAÇÃO LOCAL (1-2 horas)

#### 0.1. Correções de Segurança Críticas
**Status:** 🔴 OBRIGATÓRIO ANTES DO DEPLOY

```bash
# No seu ambiente local/desenvolvimento:

# 1. Criar novo arquivo .env.production (NÃO commitar!)
cp .env.example .env.production

# Editar .env.production com valores de produção:
nano .env.production
```

Configurações obrigatórias em `.env.production`:
```env
APP_NAME=Freefi
APP_ENV=production
APP_DEBUG=false                    # ← CRÍTICO: false em produção
APP_URL=https://SEU_DOMINIO.com.br

DB_CONNECTION=mysql
DB_HOST=db                         # ou IP do banco gerenciado
DB_PORT=3306
DB_DATABASE=adm_freefi
DB_USERNAME=NOVO_USUARIO           # ← Trocar credenciais
DB_PASSWORD=NOVA_SENHA_SEGURA      # ← Trocar credenciais

SESSION_DRIVER=database
QUEUE_CONNECTION=database
CACHE_DRIVER=redis                 # ← Usar Redis em produção

REDIS_HOST=redis
REDIS_PASSWORD=null
REDIS_PORT=6379

MAIL_MAILER=smtp
MAIL_HOST=email-ssl.com.br
MAIL_PORT=465
MAIL_USERNAME=NOVO_EMAIL           # ← Trocar credenciais
MAIL_PASSWORD=NOVA_SENHA_EMAIL     # ← Trocar credenciais
MAIL_ENCRYPTION=ssl
MAIL_FROM_ADDRESS=comercial@seudominio.com.br
MAIL_FROM_NAME="${APP_NAME}"

LOG_CHANNEL=stack
LOG_LEVEL=warning                  # ← info ou warning em produção
```

```bash
# 2. Garantir que .env está no .gitignore
echo ".env" >> .gitignore
echo ".env.production" >> .gitignore
git add .gitignore
git commit -m "security: Add .env to gitignore"

# 3. Remover .env do histórico do Git (se já foi commitado)
git filter-branch --force --index-filter \
  "git rm --cached --ignore-unmatch .env" \
  --prune-empty --tag-name-filter cat -- --all

# 4. Gerar nova APP_KEY
php artisan key:generate --show
# Copiar a chave gerada e adicionar ao .env.production

# 5. Push das mudanças
git push --force --all
```

#### 0.2. Implementar Rate Limiting (15 minutos)

Editar `routes/api.php`:
```php
// Adicionar no início do arquivo
use Illuminate\Support\Facades\Route;

// Endpoints públicos com rate limiting
Route::middleware('throttle:60,1')->group(function () {
    Route::get('/test', function () {
        return response()->json(['status' => 'ok']);
    });

    Route::post('hotspot/heartbeat', [App\Http\Controllers\Api\HotspotApiController::class, 'heartbeat']);
    Route::get('hotspot/{serial}/publicacoes', [App\Http\Controllers\Api\HotspotApiController::class, 'getPublicacoes']);
});

// Endpoints de tracking com limite maior
Route::middleware('throttle:100,1')->group(function () {
    Route::post('hotspot/impressao', [App\Http\Controllers\Api\HotspotApiController::class, 'impressao']);
    Route::post('hotspot/clique', [App\Http\Controllers\Api\HotspotApiController::class, 'clique']);
});

Route::get('hotspot/publicacao/{id}/walled-gardens', [App\Http\Controllers\Api\HotspotApiController::class, 'getWalledGardens']);

// Rotas protegidas (requerem autenticação Sanctum)
Route::middleware('auth:sanctum')->group(function () {
    Route::get('/user', function (Request $request) {
        return $request->user();
    });

    Route::post('publicacoes/{id}/impressao', [App\Http\Controllers\PublicacaoController::class, 'incrementarImpressao']);
    Route::post('publicacoes/{id}/clique', [App\Http\Controllers\PublicacaoController::class, 'incrementarClique']);
});
```

```bash
# Commitar mudanças
git add routes/api.php
git commit -m "security: Add rate limiting to API endpoints"
git push origin main
```

#### 0.3. Preparar Docker para Produção (30 minutos)

Criar `docker-compose.production.yml`:
```yaml
version: '3.8'

services:
  app:
    build:
      context: .
      dockerfile: Dockerfile
    container_name: freefi_app
    restart: unless-stopped
    working_dir: /var/www/html
    volumes:
      - ./:/var/www/html
      - ./storage:/var/www/html/storage
    environment:
      - APP_ENV=production
      - APP_DEBUG=false
    networks:
      - freefi-network
    depends_on:
      db:
        condition: service_healthy
      redis:
        condition: service_started

  db:
    image: mysql:8.0
    container_name: freefi_db
    restart: unless-stopped
    environment:
      MYSQL_DATABASE: ${DB_DATABASE}
      MYSQL_USER: ${DB_USERNAME}
      MYSQL_PASSWORD: ${DB_PASSWORD}
      MYSQL_ROOT_PASSWORD: ${DB_PASSWORD}
    volumes:
      - db_data:/var/lib/mysql
    networks:
      - freefi-network
    healthcheck:
      test: ["CMD", "mysqladmin", "ping", "-h", "localhost"]
      interval: 10s
      timeout: 5s
      retries: 5

  redis:
    image: redis:alpine
    container_name: freefi_redis
    restart: unless-stopped
    ports:
      - "6379:6379"
    networks:
      - freefi-network
    healthcheck:
      test: ["CMD", "redis-cli", "ping"]
      interval: 10s
      timeout: 3s
      retries: 5

  webserver:
    image: nginx:alpine
    container_name: freefi_webserver
    restart: unless-stopped
    ports:
      - "80:80"
      - "443:443"
    volumes:
      - ./:/var/www/html
      - ./docker/nginx-production.conf:/etc/nginx/conf.d/default.conf
      - ./docker/ssl:/etc/nginx/ssl
      - ./storage/app/public:/var/www/html/public/storage
    networks:
      - freefi-network
    depends_on:
      - app

  queue:
    build:
      context: .
      dockerfile: Dockerfile
    container_name: freefi_queue
    restart: unless-stopped
    working_dir: /var/www/html
    command: php artisan queue:work --sleep=3 --tries=3 --max-time=3600
    volumes:
      - ./:/var/www/html
    environment:
      - APP_ENV=production
    networks:
      - freefi-network
    depends_on:
      - db
      - redis

  scheduler:
    build:
      context: .
      dockerfile: Dockerfile
    container_name: freefi_scheduler
    restart: unless-stopped
    working_dir: /var/www/html
    command: /bin/sh -c "while true; do php artisan schedule:run --verbose --no-interaction & sleep 60; done"
    volumes:
      - ./:/var/www/html
    environment:
      - APP_ENV=production
    networks:
      - freefi-network
    depends_on:
      - db
      - redis

networks:
  freefi-network:
    driver: bridge

volumes:
  db_data:
    driver: local
```

Criar `docker/nginx-production.conf`:
```nginx
server {
    listen 80;
    server_name _;

    # Redirect to HTTPS
    return 301 https://$host$request_uri;
}

server {
    listen 443 ssl http2;
    server_name SEU_DOMINIO.com.br;

    root /var/www/html/public;
    index index.php index.html;

    # SSL Configuration (será preenchido com Let's Encrypt)
    ssl_certificate /etc/nginx/ssl/cert.pem;
    ssl_certificate_key /etc/nginx/ssl/key.pem;
    ssl_protocols TLSv1.2 TLSv1.3;
    ssl_ciphers HIGH:!aNULL:!MD5;
    ssl_prefer_server_ciphers on;

    # Security Headers
    add_header X-Frame-Options "SAMEORIGIN" always;
    add_header X-Content-Type-Options "nosniff" always;
    add_header X-XSS-Protection "1; mode=block" always;
    add_header Referrer-Policy "no-referrer-when-downgrade" always;
    add_header Strict-Transport-Security "max-age=31536000; includeSubDomains" always;

    # Gzip Compression
    gzip on;
    gzip_vary on;
    gzip_proxied any;
    gzip_comp_level 6;
    gzip_types text/plain text/css text/xml text/javascript application/json application/javascript application/xml+rss application/rss+xml font/truetype font/opentype application/vnd.ms-fontobject image/svg+xml;

    # Client max body size
    client_max_body_size 100M;

    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }

    location ~ \.php$ {
        fastcgi_pass app:9000;
        fastcgi_index index.php;
        fastcgi_param SCRIPT_FILENAME $document_root$fastcgi_script_name;
        include fastcgi_params;
        fastcgi_read_timeout 300;
    }

    location /storage {
        alias /var/www/html/public/storage;
        access_log off;
        expires max;
    }

    location ~ /\.(?!well-known).* {
        deny all;
    }

    access_log /var/log/nginx/access.log;
    error_log /var/log/nginx/error.log;
}
```

```bash
# Commitar arquivos de produção
git add docker-compose.production.yml docker/nginx-production.conf
git commit -m "deploy: Add production Docker configuration"
git push origin main
```

---

### FASE 1: PREPARAÇÃO DO SERVIDOR LINODE (1-2 horas)

#### 1.1. Acesso Inicial e Atualização do Sistema

```bash
# SSH no servidor Linode
ssh root@SEU_IP_LINODE

# Atualizar sistema
apt update && apt upgrade -y

# Instalar pacotes essenciais
apt install -y curl git vim ufw fail2ban htop net-tools
```

#### 1.2. Criar Usuário de Deploy (Segurança)

```bash
# Criar usuário (não usar root para deploy)
adduser deploy
usermod -aG sudo deploy

# Configurar SSH para o usuário deploy
mkdir -p /home/deploy/.ssh
chmod 700 /home/deploy/.ssh

# Copiar chave SSH pública para o servidor
# No seu computador local:
cat ~/.ssh/id_rsa.pub

# No servidor, adicionar a chave:
echo "SUA_CHAVE_PUBLICA_AQUI" >> /home/deploy/.ssh/authorized_keys
chmod 600 /home/deploy/.ssh/authorized_keys
chown -R deploy:deploy /home/deploy/.ssh

# Testar login com novo usuário
# (em outro terminal)
ssh deploy@SEU_IP_LINODE
```

#### 1.3. Configurar Firewall (UFW)

```bash
# Voltar como root ou usar sudo
sudo ufw default deny incoming
sudo ufw default allow outgoing
sudo ufw allow ssh
sudo ufw allow 80/tcp
sudo ufw allow 443/tcp
sudo ufw enable

# Verificar status
sudo ufw status verbose
```

#### 1.4. Instalar Docker e Docker Compose

```bash
# Instalar Docker
curl -fsSL https://get.docker.com -o get-docker.sh
sudo sh get-docker.sh

# Adicionar usuário deploy ao grupo docker
sudo usermod -aG docker deploy

# Instalar Docker Compose V2
sudo apt install docker-compose-plugin -y

# Verificar instalação
docker --version
docker compose version

# Logout e login novamente para aplicar grupo docker
exit
ssh deploy@SEU_IP_LINODE
```

#### 1.5. Configurar Swap (Opcional mas Recomendado)

```bash
# Se o servidor tiver menos de 4GB RAM, criar swap de 4GB
sudo fallocate -l 4G /swapfile
sudo chmod 600 /swapfile
sudo mkswap /swapfile
sudo swapon /swapfile

# Tornar permanente
echo '/swapfile none swap sw 0 0' | sudo tee -a /etc/fstab

# Verificar
free -h
```

---

### FASE 2: CONFIGURAÇÃO DO PROJETO NO SERVIDOR (1 hora)

#### 2.1. Clonar Repositório

```bash
# Como usuário deploy
cd /var/www
sudo mkdir -p freefi
sudo chown deploy:deploy freefi
cd freefi

# Clonar repositório
git clone git@github.com:freefi-com-br/freefiadm.git .

# Ou se usar HTTPS com token:
git clone https://SEU_GITHUB_PAT@github.com/freefi-com-br/freefiadm.git .
```

#### 2.2. Configurar Variáveis de Ambiente

```bash
# Copiar .env.example para .env
cp .env.example .env

# Editar .env com valores de produção
nano .env
```

Configurar conforme o arquivo `.env.production` criado na FASE 0.

**IMPORTANTE:** Usar credenciais NOVAS e SEGURAS. Não usar as credenciais expostas no código!

```bash
# Gerar APP_KEY
docker compose run --rm app php artisan key:generate

# Verificar se a chave foi adicionada ao .env
cat .env | grep APP_KEY
```

#### 2.3. Ajustar Permissões

```bash
# Dar permissões corretas para storage e cache
sudo chown -R deploy:www-data storage bootstrap/cache
sudo chmod -R 775 storage bootstrap/cache
```

---

### FASE 3: BUILD E INICIALIZAÇÃO DOS CONTAINERS (30 minutos)

#### 3.1. Build das Imagens Docker

```bash
cd /var/www/freefi

# Build usando docker-compose.production.yml
docker compose -f docker-compose.production.yml build --no-cache

# Tempo estimado: 5-10 minutos
```

#### 3.2. Subir os Containers

```bash
# Iniciar todos os serviços
docker compose -f docker-compose.production.yml up -d

# Verificar status
docker compose -f docker-compose.production.yml ps

# Deve mostrar todos os containers rodando (healthy)
```

#### 3.3. Instalar Dependências Laravel

```bash
# Entrar no container app
docker compose -f docker-compose.production.yml exec app bash

# Dentro do container:
composer install --no-dev --optimize-autoloader
npm install
npm run build

# Sair do container
exit
```

---

### FASE 4: BANCO DE DADOS (30 minutos)

#### 4.1. Executar Migrations

```bash
# Rodar migrations
docker compose -f docker-compose.production.yml exec app php artisan migrate --force

# Responder 'yes' quando perguntado
```

#### 4.2. Seed de Dados Iniciais (Opcional)

```bash
# Se houver seeders para dados iniciais (usuário admin, etc):
docker compose -f docker-compose.production.yml exec app php artisan db:seed --force

# Ou seed específico:
docker compose -f docker-compose.production.yml exec app php artisan db:seed --class=AdminUserSeeder --force
```

#### 4.3. Criar Link de Storage

```bash
docker compose -f docker-compose.production.yml exec app php artisan storage:link
```

---

### FASE 5: CONFIGURAÇÃO SSL/HTTPS (30-60 minutos)

#### 5.1. Opção A - Let's Encrypt com Certbot (Recomendado)

```bash
# No servidor host (não no container)
sudo apt install certbot python3-certbot-nginx -y

# Parar o Nginx container temporariamente
docker compose -f docker-compose.production.yml stop webserver

# Gerar certificado
sudo certbot certonly --standalone -d SEU_DOMINIO.com.br -d www.SEU_DOMINIO.com.br

# Copiar certificados para pasta do projeto
sudo mkdir -p /var/www/freefi/docker/ssl
sudo cp /etc/letsencrypt/live/SEU_DOMINIO.com.br/fullchain.pem /var/www/freefi/docker/ssl/cert.pem
sudo cp /etc/letsencrypt/live/SEU_DOMINIO.com.br/privkey.pem /var/www/freefi/docker/ssl/key.pem
sudo chown -R deploy:deploy /var/www/freefi/docker/ssl

# Reiniciar webserver
docker compose -f docker-compose.production.yml start webserver
```

#### 5.2. Renovação Automática de Certificados

```bash
# Criar script de renovação
sudo nano /usr/local/bin/renew-ssl.sh
```

Conteúdo do script:
```bash
#!/bin/bash
docker compose -f /var/www/freefi/docker-compose.production.yml stop webserver
certbot renew --quiet
cp /etc/letsencrypt/live/SEU_DOMINIO.com.br/fullchain.pem /var/www/freefi/docker/ssl/cert.pem
cp /etc/letsencrypt/live/SEU_DOMINIO.com.br/privkey.pem /var/www/freefi/docker/ssl/key.pem
chown -R deploy:deploy /var/www/freefi/docker/ssl
docker compose -f /var/www/freefi/docker-compose.production.yml start webserver
```

```bash
# Dar permissão de execução
sudo chmod +x /usr/local/bin/renew-ssl.sh

# Adicionar ao cron (rodar todo dia às 3am)
sudo crontab -e

# Adicionar linha:
0 3 * * * /usr/local/bin/renew-ssl.sh >> /var/log/letsencrypt-renew.log 2>&1
```

---

### FASE 6: OTIMIZAÇÕES DE PRODUÇÃO (15 minutos)

#### 6.1. Cache de Configurações

```bash
docker compose -f docker-compose.production.yml exec app php artisan config:cache
docker compose -f docker-compose.production.yml exec app php artisan route:cache
docker compose -f docker-compose.production.yml exec app php artisan view:cache
```

#### 6.2. Otimizar Autoload do Composer

```bash
docker compose -f docker-compose.production.yml exec app composer dump-autoload --optimize
```

---

### FASE 7: CONFIGURAR GITHUB ACTIONS SECRETS (10 minutos)

No GitHub, ir em: **Settings > Secrets and variables > Actions > New repository secret**

Adicionar os seguintes secrets:

| Secret Name | Value |
|-------------|-------|
| `SSH_HOST` | IP do servidor Linode |
| `SSH_USERNAME` | deploy |
| `SSH_PRIVATE_KEY` | Conteúdo da chave privada SSH |
| `SSH_PORT` | 22 |
| `PROJECT_PATH` | /var/www/freefi |
| `GH_PAT` | GitHub Personal Access Token |

---

### FASE 8: TESTES DE FUNCIONAMENTO (30 minutos)

#### 8.1. Testar Acesso Web

```bash
# Testar HTTP (deve redirecionar para HTTPS)
curl -I http://SEU_DOMINIO.com.br

# Testar HTTPS
curl -I https://SEU_DOMINIO.com.br
```

Abrir no navegador: `https://SEU_DOMINIO.com.br`

#### 8.2. Testar API

```bash
# Testar endpoint de health check
curl https://SEU_DOMINIO.com.br/api/test

# Testar rate limiting (deve retornar 429 após 60 requests)
for i in {1..65}; do
  curl https://SEU_DOMINIO.com.br/api/test
  echo "Request $i"
done
```

#### 8.3. Testar Banco de Dados

```bash
# Acessar banco
docker compose -f docker-compose.production.yml exec db mysql -u root -p

# Dentro do MySQL:
SHOW DATABASES;
USE adm_freefi;
SHOW TABLES;
SELECT * FROM users LIMIT 1;
EXIT;
```

#### 8.4. Testar Upload de Arquivos

- Fazer login no sistema
- Tentar fazer upload de uma imagem
- Verificar se o arquivo foi salvo em `storage/app/public`

#### 8.5. Testar Envio de Emails

- Registrar novo usuário (se permitido)
- Verificar se email de confirmação foi enviado
- Checar logs: `docker compose -f docker-compose.production.yml logs app | grep mail`

#### 8.6. Testar Queue

```bash
# Verificar se queue worker está rodando
docker compose -f docker-compose.production.yml ps queue

# Ver logs do queue
docker compose -f docker-compose.production.yml logs -f queue

# Processar job de teste (se houver)
docker compose -f docker-compose.production.yml exec app php artisan queue:work --once
```

---

### FASE 9: MONITORAMENTO E LOGS (15 minutos)

#### 9.1. Configurar Logs Persistentes

```bash
# Criar diretório de logs
mkdir -p /var/www/freefi/storage/logs

# Configurar logrotate
sudo nano /etc/logrotate.d/freefi
```

Conteúdo:
```
/var/www/freefi/storage/logs/*.log {
    daily
    missingok
    rotate 14
    compress
    delaycompress
    notifempty
    create 0640 deploy deploy
    sharedscripts
}
```

#### 9.2. Monitoramento de Containers

```bash
# Instalar ctop (dashboard de containers)
sudo wget https://github.com/bcicen/ctop/releases/download/v0.7.7/ctop-0.7.7-linux-amd64 -O /usr/local/bin/ctop
sudo chmod +x /usr/local/bin/ctop

# Executar:
ctop
```

#### 9.3. Script de Health Check

```bash
nano /home/deploy/health-check.sh
```

Conteúdo:
```bash
#!/bin/bash
SITE="https://SEU_DOMINIO.com.br"
EMAIL="seu-email@dominio.com"

if ! curl -f -s -o /dev/null "$SITE"; then
    echo "Site $SITE está DOWN!" | mail -s "ALERTA: Site Down" $EMAIL

    # Tentar reiniciar containers
    cd /var/www/freefi
    docker compose -f docker-compose.production.yml restart
fi
```

```bash
chmod +x /home/deploy/health-check.sh

# Agendar verificação a cada 5 minutos
crontab -e
# Adicionar:
*/5 * * * * /home/deploy/health-check.sh
```

---

### FASE 10: BACKUP AUTOMÁTICO (30 minutos)

#### 10.1. Script de Backup

```bash
sudo mkdir -p /backups
sudo chown deploy:deploy /backups

nano /home/deploy/backup.sh
```

Conteúdo:
```bash
#!/bin/bash
set -e

DATE=$(date +%Y%m%d_%H%M%S)
BACKUP_DIR="/backups"
PROJECT_DIR="/var/www/freefi"

echo "=== Iniciando backup em $DATE ==="

# Backup do banco de dados
echo "Fazendo backup do banco..."
docker compose -f $PROJECT_DIR/docker-compose.production.yml exec -T db \
  mysqldump -u root -p${DB_PASSWORD} adm_freefi \
  > $BACKUP_DIR/db_backup_$DATE.sql

# Compactar backup do banco
gzip $BACKUP_DIR/db_backup_$DATE.sql

# Backup do diretório storage
echo "Fazendo backup do storage..."
tar -czf $BACKUP_DIR/storage_backup_$DATE.tar.gz -C $PROJECT_DIR storage/

# Backup do .env
echo "Fazendo backup do .env..."
cp $PROJECT_DIR/.env $BACKUP_DIR/env_backup_$DATE

# Listar backups
echo "Backups criados:"
ls -lh $BACKUP_DIR/*$DATE*

# Deletar backups com mais de 7 dias
echo "Limpando backups antigos..."
find $BACKUP_DIR -name "db_backup_*.sql.gz" -mtime +7 -delete
find $BACKUP_DIR -name "storage_backup_*.tar.gz" -mtime +7 -delete
find $BACKUP_DIR -name "env_backup_*" -mtime +7 -delete

# Enviar para armazenamento remoto (opcional)
# Exemplos:
# rsync -avz $BACKUP_DIR/ usuario@backup-server:/backups/freefi/
# aws s3 sync $BACKUP_DIR s3://meu-bucket-backups/freefi/
# rclone sync $BACKUP_DIR remote:freefi-backups

echo "=== Backup concluído! ==="
```

```bash
chmod +x /home/deploy/backup.sh

# Agendar backup diário às 2am
crontab -e
# Adicionar:
0 2 * * * /home/deploy/backup.sh >> /var/log/freefi-backup.log 2>&1
```

#### 10.2. Testar Backup

```bash
# Executar manualmente
/home/deploy/backup.sh

# Verificar se arquivos foram criados
ls -lh /backups/
```

#### 10.3. Testar Restore (IMPORTANTE!)

```bash
# Simular restore do banco
docker compose -f /var/www/freefi/docker-compose.production.yml exec -T db \
  mysql -u root -p${DB_PASSWORD} adm_freefi \
  < /backups/db_backup_YYYYMMDD_HHMMSS.sql

# Se der erro, ajustar script de backup
```

---

## 📊 CHECKLIST FINAL PRÉ-DEPLOY

Antes de considerar o deploy concluído, verificar:

### Segurança
- [ ] APP_DEBUG=false no .env de produção
- [ ] .env não está no repositório Git
- [ ] Credenciais rotacionadas (não usar as expostas)
- [ ] Firewall configurado (UFW)
- [ ] Fail2ban instalado e ativo
- [ ] SSH configurado apenas com chave (sem senha)
- [ ] SSL/HTTPS funcionando
- [ ] Headers de segurança configurados no Nginx
- [ ] Rate limiting testado nas APIs

### Aplicação
- [ ] Migrations executadas sem erros
- [ ] Storage link criado
- [ ] Permissões de arquivos corretas (775 em storage)
- [ ] Config/Route/View cache criado
- [ ] Composer otimizado (--no-dev)
- [ ] Assets compilados (npm run build)

### Infraestrutura
- [ ] Todos os containers rodando (docker ps)
- [ ] Banco de dados acessível
- [ ] Redis funcionando
- [ ] Queue worker ativo
- [ ] Scheduler ativo
- [ ] Nginx servindo arquivos estáticos

### Testes
- [ ] Homepage carrega corretamente
- [ ] Login funciona
- [ ] Upload de arquivos funciona
- [ ] API responde corretamente
- [ ] Emails sendo enviados (ou queueados)
- [ ] Rate limiting funciona (429 após limite)

### Monitoramento
- [ ] Logs configurados
- [ ] Health check script ativo
- [ ] Backups automáticos agendados
- [ ] Backup testado e restore validado
- [ ] SSL renovação automática configurada

### CI/CD
- [ ] GitHub Actions secrets configurados
- [ ] Pipeline de deploy testada (push para main)
- [ ] Rollback testado

---

## 🚨 PROCEDIMENTO DE EMERGÊNCIA - ROLLBACK

Se algo der errado durante ou após o deploy:

### Rollback Rápido (Código)
```bash
cd /var/www/freefi

# Voltar para commit anterior
git log --oneline -n 10
git checkout <COMMIT_HASH_ANTERIOR>

# Rebuild e restart
docker compose -f docker-compose.production.yml down
docker compose -f docker-compose.production.yml up -d --build

# Limpar caches
docker compose -f docker-compose.production.yml exec app php artisan cache:clear
docker compose -f docker-compose.production.yml exec app php artisan config:clear
```

### Rollback de Banco de Dados
```bash
# CUIDADO: Pode causar perda de dados!

# Parar aplicação
docker compose -f docker-compose.production.yml stop app webserver queue

# Restaurar backup
docker compose -f docker-compose.production.yml exec -T db \
  mysql -u root -p${DB_PASSWORD} adm_freefi \
  < /backups/db_backup_YYYYMMDD_HHMMSS.sql

# Reiniciar
docker compose -f docker-compose.production.yml start app webserver queue
```

### Página de Manutenção
```bash
# Ativar modo manutenção
docker compose -f docker-compose.production.yml exec app php artisan down \
  --message="Manutenção programada. Voltamos em breve."

# Executar correções...

# Desativar
docker compose -f docker-compose.production.yml exec app php artisan up
```

---

## 📞 SUPORTE E PRÓXIMOS PASSOS

### Após Deploy Bem-Sucedido

1. **Monitorar por 24-48h:**
   - Verificar logs: `docker compose logs -f`
   - Verificar uso de recursos: `htop`, `ctop`
   - Testar todas as funcionalidades principais

2. **Configurar Monitoring Externo:**
   - UptimeRobot (free): https://uptimerobot.com
   - Pingdom
   - StatusCake

3. **Implementar as Fases 2-6 do arquivo analise-cloudeia.txt:**
   - Autorização com Policies
   - Criptografia de IPs
   - Testes de segurança
   - Documentação da API

4. **Considerar Melhorias:**
   - CDN (Cloudflare)
   - APM (New Relic, DataDog)
   - Error tracking (Sentry)
   - Log aggregation (Papertrail)

### Comandos Úteis de Manutenção

```bash
# Ver logs em tempo real
docker compose -f docker-compose.production.yml logs -f app

# Reiniciar um serviço específico
docker compose -f docker-compose.production.yml restart app

# Entrar no container para debug
docker compose -f docker-compose.production.yml exec app bash

# Limpar cache Laravel
docker compose -f docker-compose.production.yml exec app php artisan cache:clear
docker compose -f docker-compose.production.yml exec app php artisan config:clear
docker compose -f docker-compose.production.yml exec app php artisan view:clear

# Ver uso de disco
df -h
du -sh /var/www/freefi/*

# Ver uso de memória/CPU
htop
ctop

# Atualizar código (manual)
cd /var/www/freefi
git pull origin main
docker compose -f docker-compose.production.yml exec app composer install --no-dev
docker compose -f docker-compose.production.yml exec app php artisan migrate --force
docker compose -f docker-compose.production.yml restart
```

---

## 📋 INFORMAÇÕES ADICIONAIS

### Análise da Pasta `/var/www/html/laravel/`

**Conclusão:** ✅ **A pasta `/var/www/html/laravel/` PODE SER REMOVIDA com segurança**

**Justificativa:**
1. Seu projeto principal está em: `/var/www/html/laravel/admin/freefi_admin/`
2. O repositório Git correto está dentro dessa pasta
3. A pasta `/var/www/html/laravel/` contém:
   - **freefi_admin_bk** (371MB) - Backup antigo
   - **laravel_flv_mvp_rbac_dynamic** (390MB) - Outro projeto
   - **ki-admin_laravel** (317MB) - Template comprado
   - Projetos de outros sistemas (clinical, cafe, etc)

**Recomendações:**
1. Mover o projeto freefi_admin para `/var/www/freefi/` diretamente
2. Fazer backup da pasta laravel antes de remover
3. Remover a pasta laravel após confirmar que tudo está funcionando

**Comando para reorganizar:**
```bash
# Fazer backup da pasta laravel inteira
tar -czf /backups/laravel_backup_$(date +%Y%m%d).tar.gz /var/www/html/laravel/

# Mover projeto principal para local limpo
sudo mv /var/www/html/laravel/admin/freefi_admin /var/www/freefi
sudo chown -R deploy:deploy /var/www/freefi

# Após confirmar funcionamento, remover pasta antiga
sudo rm -rf /var/www/html/laravel/
```

---

## ✅ STATUS DO PLANO

**Este documento está pronto para execução assim que você fornecer:**

1. ✅ Credenciais SSH do servidor Linode
2. ✅ Informações do servidor (IP, OS, recursos)
3. ✅ Domínio e configuração DNS
4. ✅ Credenciais de email SMTP (novas, não as expostas)
5. ✅ Decisão sobre banco de dados (Docker ou gerenciado)

**Tempo total estimado:** 6-8 horas (incluindo testes)

**Quando estiver pronto, me envie as credenciais e vamos começar o deploy!**

---

**Documento criado por:** Cloudeia AI
**Data:** 21/10/2025
**Versão:** 1.0
