# 📡 COMO FUNCIONA O TIPO DE ALCANCE - MÓDULO DE PUBLICAÇÕES

## 📋 Visão Geral

O **Tipo de Alcance** determina **onde** (em quais equipamentos) uma publicação será exibida. É um sistema de **filtragem inteligente** com 3 opções.

---

## 🎯 AS 3 OPÇÕES DE ALCANCE

### 1️⃣ **TODOS OS EQUIPAMENTOS** (padrão)
```
tipo_alcance = 'todos'
```

**O que faz:**
- A publicação será exibida em **TODOS** os equipamentos cadastrados
- Não precisa especificar serial ou empresa
- É a opção mais ampla

**Exemplo de Uso:**
- Campanhas nacionais
- Promoções gerais da rede
- Avisos importantes para todos

**No Banco de Dados:**
```php
tipo_alcance: 'todos'
equipment_serial: NULL
empresa_alcance_id: NULL
```

---

### 2️⃣ **SERIAL ESPECÍFICO**
```
tipo_alcance = 'serial_especifico'
equipment_serial = 'ABC123456'
```

**O que faz:**
- A publicação será exibida em **APENAS UM** equipamento específico
- Você precisa informar o **serial do equipamento**
- É a opção mais restrita

**Exemplo de Uso:**
- Teste de campanha em equipamento específico
- Promoção local de uma loja
- Conteúdo personalizado para um ponto de venda

**No Banco de Dados:**
```php
tipo_alcance: 'serial_especifico'
equipment_serial: 'ABC123456'  // Serial do equipamento
empresa_alcance_id: NULL
```

---

### 3️⃣ **EMPRESA ESPECÍFICA**
```
tipo_alcance = 'empresa'
empresa_alcance_id = 5
```

**O que faz:**
- A publicação será exibida em **TODOS os equipamentos** de uma empresa
- Você precisa selecionar qual **empresa** (cliente)
- É a opção intermediária

**Exemplo de Uso:**
- Campanha para todos os equipamentos de um Shopping
- Promoção para toda rede de lojas de uma marca
- Conteúdo específico para franquia

**No Banco de Dados:**
```php
tipo_alcance: 'empresa'
equipment_serial: NULL
empresa_alcance_id: 5  // ID da empresa
```

---

## 🎬 FUNCIONAMENTO NA INTERFACE (Frontend)

### **HTML - Select Principal**
```blade
<select name="tipo_alcance" id="tipo_alcance" required>
    <option value="todos">Todos os equipamentos</option>
    <option value="serial_especifico">Serial Específico</option>
    <option value="empresa">Empresa Específica</option>
</select>
```

### **Campos Condicionais (ocultos por padrão)**

#### Campo 1: Serial do Equipamento
```blade
<div id="serial_field" style="display: none;">
    <input type="text" name="equipment_serial" placeholder="Ex: ABC123456">
</div>
```
- **Aparece quando:** tipo_alcance = 'serial_especifico'
- **Validação:** required_if:tipo_alcance,serial_especifico

#### Campo 2: Empresa Específica
```blade
<div id="empresa_field" style="display: none;">
    <select name="empresa_alcance_id">
        <option value="">Selecione uma empresa</option>
        @foreach($clientes as $cliente)
            <option value="{{ $cliente->id }}">{{ $cliente->name }}</option>
        @endforeach
    </select>
</div>
```
- **Aparece quando:** tipo_alcance = 'empresa'
- **Validação:** required_if:tipo_alcance,empresa

---

## ⚙️ LÓGICA JAVASCRIPT

### **Controle de Exibição Dinâmica**

```javascript
document.addEventListener('DOMContentLoaded', function() {
    const tipoAlcanceSelect = document.getElementById('tipo_alcance');
    const serialField = document.getElementById('serial_field');
    const empresaField = document.getElementById('empresa_field');
    
    function updateFields() {
        const value = tipoAlcanceSelect.value;
        
        // 1. Ocultar TODOS os campos primeiro
        serialField.style.display = 'none';
        empresaField.style.display = 'none';
        
        // 2. Mostrar apenas o campo relevante
        if (value === 'serial_especifico') {
            serialField.style.display = 'block';
        } else if (value === 'empresa') {
            empresaField.style.display = 'block';
        }
        // Se value === 'todos', ambos ficam ocultos
    }
    
    // Executar ao carregar a página
    updateFields();
    
    // Executar quando o usuário mudar o select
    tipoAlcanceSelect.addEventListener('change', updateFields);
});
```

### **Fluxo de Interação:**

1. **Usuário abre formulário** → JavaScript executa `updateFields()`
2. **Select está em "todos"** → Nenhum campo extra aparece
3. **Usuário muda para "Serial Específico"** → Campo de serial aparece
4. **Usuário muda para "Empresa Específica"** → Campo de empresa aparece
5. **Usuário volta para "todos"** → Ambos desaparecem

---

## 🔒 VALIDAÇÃO NO BACKEND (Controller)

### **Regras de Validação:**

```php
$validated = $request->validate([
    // Campo obrigatório
    'tipo_alcance' => 'required|in:todos,serial_especifico,empresa',
    
    // Condicional: obrigatório SE tipo_alcance = 'serial_especifico'
    'equipment_serial' => 'nullable|string|max:255|required_if:tipo_alcance,serial_especifico',
    
    // Condicional: obrigatório SE tipo_alcance = 'empresa'
    'empresa_alcance_id' => 'nullable|integer|exists:empresas,id|required_if:tipo_alcance,empresa',
]);
```

### **Cenários de Validação:**

| Tipo Alcance | equipment_serial | empresa_alcance_id | Resultado |
|--------------|------------------|-------------------|-----------|
| `todos` | NULL | NULL | ✅ VÁLIDO |
| `serial_especifico` | 'ABC123' | NULL | ✅ VÁLIDO |
| `serial_especifico` | NULL | NULL | ❌ ERRO (serial obrigatório) |
| `empresa` | NULL | 5 | ✅ VÁLIDO |
| `empresa` | NULL | NULL | ❌ ERRO (empresa obrigatória) |
| `empresa` | 'ABC123' | 5 | ✅ VÁLIDO (serial é ignorado) |

---

## 🗄️ ESTRUTURA DO BANCO DE DADOS

### **Tabela: publicacoes**

```sql
CREATE TABLE publicacoes (
    id BIGINT PRIMARY KEY,
    tipo_alcance ENUM('todos', 'serial_especifico', 'empresa') DEFAULT 'todos',
    equipment_serial VARCHAR(255) NULL,
    empresa_alcance_id BIGINT NULL,
    -- outros campos...
    
    FOREIGN KEY (empresa_alcance_id) REFERENCES empresas(id)
);
```

### **Exemplos de Registros:**

```sql
-- Publicação para TODOS
INSERT INTO publicacoes (tipo_alcance, equipment_serial, empresa_alcance_id)
VALUES ('todos', NULL, NULL);

-- Publicação para SERIAL específico
INSERT INTO publicacoes (tipo_alcance, equipment_serial, empresa_alcance_id)
VALUES ('serial_especifico', 'ABC123456', NULL);

-- Publicação para EMPRESA específica
INSERT INTO publicacoes (tipo_alcance, equipment_serial, empresa_alcance_id)
VALUES ('empresa', NULL, 5);
```

---

## 🎯 LÓGICA DE APLICAÇÃO (Model)

### **Método: aplicavelParaEquipamento()**

Este método verifica se uma publicação deve ser exibida em um equipamento específico:

```php
public function aplicavelParaEquipamento($serial, $empresaId = null): bool
{
    // CASO 1: Todos os equipamentos
    if ($this->tipo_alcance === 'todos') {
        return true;  // ✅ Sempre exibe
    }
    
    // CASO 2: Serial específico
    if ($this->tipo_alcance === 'serial_especifico') {
        return $this->equipment_serial === $serial;  // ✅ Só se o serial combinar
    }
    
    // CASO 3: Empresa específica
    if ($this->tipo_alcance === 'empresa' && $empresaId) {
        return $this->empresa_alcance_id == $empresaId;  // ✅ Só se a empresa combinar
    }
    
    return false;  // ❌ Não exibe
}
```

### **Exemplos de Uso:**

```php
// Equipamento: Serial = 'ABC123', Empresa ID = 5

$pub1 = Publicacao::find(1);  // tipo_alcance = 'todos'
$pub1->aplicavelParaEquipamento('ABC123', 5);  // ✅ true

$pub2 = Publicacao::find(2);  // tipo_alcance = 'serial_especifico', equipment_serial = 'ABC123'
$pub2->aplicavelParaEquipamento('ABC123', 5);  // ✅ true
$pub2->aplicavelParaEquipamento('XYZ999', 5);  // ❌ false

$pub3 = Publicacao::find(3);  // tipo_alcance = 'empresa', empresa_alcance_id = 5
$pub3->aplicavelParaEquipamento('ABC123', 5);  // ✅ true
$pub3->aplicavelParaEquipamento('ABC123', 7);  // ❌ false
```

---

## 📊 SCOPES DISPONÍVEIS

### **1. ParaTodos()**
```php
Publicacao::paraTodos()->get();
// SELECT * FROM publicacoes WHERE tipo_alcance = 'todos'
```

### **2. SerialEspecifico($serial)**
```php
Publicacao::serialEspecifico('ABC123')->get();
// SELECT * FROM publicacoes WHERE tipo_alcance = 'serial_especifico' AND equipment_serial = 'ABC123'
```

### **3. ParaEmpresa($empresaId)**
```php
Publicacao::paraEmpresa(5)->get();
// SELECT * FROM publicacoes WHERE tipo_alcance = 'empresa' AND empresa_alcance_id = 5
```

---

## 🎨 EXIBIÇÃO NA LISTAGEM

### **Badge de Alcance:**

```blade
@php
    $alcanceBadges = [
        'todos' => ['class' => 'bg-primary', 'icon' => 'ph-globe'],
        'serial_especifico' => ['class' => 'bg-warning', 'icon' => 'ph-barcode'],
        'empresa' => ['class' => 'bg-info', 'icon' => 'ph-buildings'],
    ];
    $badge = $alcanceBadges[$publicacao->tipo_alcance] ?? ['class' => 'bg-secondary', 'icon' => 'ph-question'];
@endphp

<span class="badge {{ $badge['class'] }}">
    <i class="ph-duotone {{ $badge['icon'] }} me-1"></i>
    {{ $publicacao->getTipoAlcanceLabel() }}
</span>

@if($publicacao->tipo_alcance === 'serial_especifico')
    <br><small class="text-muted">{{ $publicacao->equipment_serial }}</small>
@elseif($publicacao->tipo_alcance === 'empresa' && $publicacao->empresaAlcance)
    <br><small class="text-muted">{{ $publicacao->empresaAlcance->name }}</small>
@endif
```

### **Resultado Visual:**

```
🌐 Todos os equipamentos  (badge azul)

📊 Serial específico       (badge amarelo)
   ABC123456

🏢 Empresa específica      (badge ciano)
   Shopping Iguatemi
```

---

## 🔄 FLUXO COMPLETO

### **1. CRIAÇÃO:**
```
Usuário → Seleciona tipo → Campo aparece → Preenche → Salva → Validação → Banco
```

### **2. CONSULTA (API/Equipamento):**
```
Equipamento ABC123 (Empresa ID 5) solicita publicações
     ↓
Query busca publicações:
  ✅ tipo_alcance = 'todos' (todas)
  ✅ tipo_alcance = 'serial_especifico' AND equipment_serial = 'ABC123'
  ✅ tipo_alcance = 'empresa' AND empresa_alcance_id = 5
     ↓
Retorna lista de publicações aplicáveis
```

### **3. EXIBIÇÃO:**
```
Equipamento → Recebe lista → Exibe criativos no horário correto
```

---

## 💡 CASOS DE USO REAIS

### **Caso 1: Campanha Nacional**
- **Alcance:** Todos os equipamentos
- **Exemplo:** Black Friday da rede
- **Equipamentos:** Todos (1000+)

### **Caso 2: Teste A/B**
- **Alcance:** Serial específico
- **Exemplo:** Testar novo criativo em 1 equipamento
- **Equipamentos:** 1 único (ABC123)

### **Caso 3: Promoção Local**
- **Alcance:** Empresa específica
- **Exemplo:** Promoção no Shopping Iguatemi
- **Equipamentos:** Todos do shopping (15 equipamentos)

---

## ⚠️ OBSERVAÇÕES IMPORTANTES

1. **Validação Condicional:** Campos são obrigatórios apenas quando relevantes
2. **Performance:** Use scopes para queries eficientes
3. **Flexibilidade:** Sistema permite migrar de "todos" → "empresa" facilmente
4. **Rastreamento:** Campo `publicada_por` registra quem criou
5. **Soft Delete:** Publicações são arquivadas, não deletadas

---

## 🎯 RESUMO RÁPIDO

| Tipo | Quando Usar | Campo Extra | Query |
|------|-------------|-------------|-------|
| **Todos** | Campanha geral | Nenhum | Sem filtro |
| **Serial** | Teste/Local | equipment_serial | WHERE serial = ? |
| **Empresa** | Rede de lojas | empresa_alcance_id | WHERE empresa = ? |

---

**Criado em:** 18/10/2025  
**Sistema:** FreeFi Admin - Módulo de Publicações  
**Versão:** 1.0
