<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Publicacao;
use Carbon\Carbon;

class ExpirePublicacoes extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'publicacoes:expire
                            {--dry-run : Execute sem fazer alterações no banco}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Marca publicações como expiradas quando a data_fim é menor que hoje';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🔍 Procurando publicações expiradas...');
        $this->newLine();

        $hoje = Carbon::today();
        
        // Buscar publicações ativas que já expiraram
        $publicacoesExpiradas = Publicacao::where('status', 'ativa')
            ->where('data_fim', '<', $hoje)
            ->get();

        if ($publicacoesExpiradas->isEmpty()) {
            $this->info('✅ Nenhuma publicação expirada encontrada!');
            return Command::SUCCESS;
        }

        $this->warn("⚠️  Encontradas {$publicacoesExpiradas->count()} publicações expiradas:");
        $this->newLine();

        $table = [];
        foreach ($publicacoesExpiradas as $pub) {
            $table[] = [
                'ID' => $pub->id,
                'Título' => $pub->titulo,
                'Cliente' => $pub->cliente?->name ?? '-',
                'Data Fim' => $pub->data_fim->format('d/m/Y'),
                'Dias Atrasado' => $hoje->diffInDays($pub->data_fim),
            ];
        }

        $this->table(['ID', 'Título', 'Cliente', 'Data Fim', 'Dias Atrasado'], $table);
        $this->newLine();

        if ($this->option('dry-run')) {
            $this->warn('🔸 Modo DRY-RUN: Nenhuma alteração será feita no banco de dados.');
            return Command::SUCCESS;
        }

        // Confirmar antes de atualizar
        if (!$this->confirm('Deseja marcar essas publicações como expiradas?', true)) {
            $this->info('Operação cancelada.');
            return Command::SUCCESS;
        }

        // Atualizar status
        $updated = Publicacao::where('status', 'ativa')
            ->where('data_fim', '<', $hoje)
            ->update(['status' => 'expirada']);

        $this->newLine();
        $this->info("✅ {$updated} publicações marcadas como expiradas!");
        
        return Command::SUCCESS;
    }
}
