<?php

namespace App\Console\Commands;

use App\Models\Empresa;
use Illuminate\Console\Command;
use Illuminate\Support\Str;

class GerarCodigosConvite extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'empresas:gerar-codigos-convite 
                            {--force : Regenerar códigos mesmo para empresas que já possuem}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Gera códigos de convite únicos para empresas sem código';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🔄 Gerando códigos de convite para empresas...');
        $this->newLine();

        // Buscar empresas sem código OU forçar regeneração
        $query = Empresa::query();
        
        if (!$this->option('force')) {
            $query->whereNull('codigo_convite');
        }
        
        $empresas = $query->get();

        if ($empresas->isEmpty()) {
            $this->warn('✅ Todas as empresas já possuem código de convite!');
            return Command::SUCCESS;
        }

        $this->info("📊 Encontradas {$empresas->count()} empresas sem código.");
        $this->newLine();

        $bar = $this->output->createProgressBar($empresas->count());
        $bar->start();

        $gerados = 0;
        $erros = 0;

        foreach ($empresas as $empresa) {
            try {
                $codigoGerado = $this->gerarCodigoUnico();
                
                $empresa->update(['codigo_convite' => $codigoGerado]);
                
                $gerados++;
                $bar->advance();
                
            } catch (\Exception $e) {
                $erros++;
                $this->newLine();
                $this->error("❌ Erro ao gerar código para empresa ID {$empresa->id}: {$e->getMessage()}");
            }
        }

        $bar->finish();
        $this->newLine(2);

        // Exibir resultados
        $this->info("✅ Códigos gerados com sucesso: {$gerados}");
        
        if ($erros > 0) {
            $this->warn("⚠️  Erros: {$erros}");
        }

        $this->newLine();
        $this->info('📋 Exemplos de códigos gerados:');
        $this->table(
            ['ID', 'Nome da Empresa', 'Código de Convite'],
            Empresa::latest('updated_at')
                ->limit(5)
                ->get(['id', 'name', 'codigo_convite'])
                ->map(fn($e) => [$e->id, $e->name, $e->codigo_convite])
        );

        return Command::SUCCESS;
    }

    /**
     * Gerar código de convite único
     * Formato: EGU2160261 (10 caracteres alfanuméricos)
     * Exemplo: ABC1234567, XYZ9876543
     */
    private function gerarCodigoUnico(): string
    {
        $tentativas = 0;
        $maxTentativas = 100;

        do {
            $codigo = $this->gerarCodigo();
            $tentativas++;
            
            if ($tentativas >= $maxTentativas) {
                throw new \RuntimeException('Não foi possível gerar código único após ' . $maxTentativas . ' tentativas');
            }
            
        } while (Empresa::where('codigo_convite', $codigo)->exists());

        return $codigo;
    }

    /**
     * Gerar código aleatório (10 caracteres alfanuméricos)
     * 3 letras maiúsculas + 7 números
     */
    private function gerarCodigo(): string
    {
        $letras = strtoupper(Str::random(3)); // ABC, XYZ, EGU
        $numeros = str_pad(rand(0, 9999999), 7, '0', STR_PAD_LEFT); // 2160261
        
        return $letras . $numeros; // EGU2160261
    }
}
