<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Hotspot;
use App\Models\Publicacao;
use App\Models\WalledGarden;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class HotspotApiController extends Controller
{
    /**
     * Heartbeat - Hotspot informa que está online
     * 
     * POST /api/hotspot/heartbeat
     * Body: { "serial": "ABC123" }
     */
    public function heartbeat(Request $request)
    {
        $validated = $request->validate([
            'serial' => 'required|string|max:255',
            'vpn_ip' => 'nullable|ip',
        ]);

        $hotspot = Hotspot::where('serial', $validated['serial'])->first();

        if (!$hotspot) {
            return response()->json([
                'success' => false,
                'message' => 'Hotspot não encontrado.',
            ], 404);
        }

        // Atualizar last_activity_at
        $hotspot->last_activity_at = now();
        
        if (isset($validated['vpn_ip'])) {
            $hotspot->vpn_ip = $validated['vpn_ip'];
        }
        
        $hotspot->save();

        return response()->json([
            'success' => true,
            'message' => 'Heartbeat registrado.',
            'data' => [
                'hotspot_id' => $hotspot->id,
                'serial' => $hotspot->serial,
                'status' => $hotspot->status ? 'ativo' : 'inativo',
                'last_activity' => $hotspot->last_activity_at->toIso8601String(),
            ]
        ]);
    }

    /**
     * Buscar publicações ativas para o hotspot
     * 
     * GET /api/hotspot/{serial}/publicacoes
     */
    public function publicacoes(Request $request, string $serial)
    {
        $hotspot = Hotspot::where('serial', $serial)->first();

        if (!$hotspot) {
            return response()->json([
                'success' => false,
                'message' => 'Hotspot não encontrado.',
            ], 404);
        }

        // Buscar publicações vigentes aplicáveis ao hotspot
        $publicacoes = Publicacao::with(['criativo', 'walledGardens'])
            ->vigentes()
            ->get()
            ->filter(function ($pub) use ($hotspot) {
                return $pub->aplicavelParaEquipamento($hotspot->serial, $hotspot->cliente_id);
            })
            ->map(function ($pub) {
                return [
                    'id' => $pub->id,
                    'titulo' => $pub->titulo,
                    'posicao' => $pub->posicao,
                    'data_inicio' => $pub->data_inicio->toDateString(),
                    'data_fim' => $pub->data_fim->toDateString(),
                    'link_destino' => $pub->link_destino,
                    'habilitar_clique' => $pub->habilitar_clique,
                    'track_impressoes' => $pub->track_impressoes,
                    'track_cliques' => $pub->track_cliques,
                    'criativo' => [
                        'id' => $pub->criativo->id,
                        'title' => $pub->criativo->title,
                        'tipo' => $pub->criativo->tipo,
                        'formato' => $pub->criativo->formato,
                        'path' => $pub->criativo->path ? url('storage/' . $pub->criativo->path) : null,
                        'conteudo_html' => $pub->criativo->conteudo_html,
                    ],
                    'walled_gardens' => $pub->walledGardens->where('ativo', true)->map(function ($wg) {
                        return [
                            'tipo' => $wg->tipo,
                            'endereco' => $wg->endereco,
                            'descricao' => $wg->descricao,
                        ];
                    })->values(),
                ];
            })
            ->values();

        return response()->json([
            'success' => true,
            'count' => $publicacoes->count(),
            'data' => $publicacoes,
        ]);
    }

    /**
     * Buscar walled gardens de uma publicação específica
     * 
     * GET /api/hotspot/publicacao/{id}/walled-gardens
     */
    public function walledGardens($publicacaoId)
    {
        $publicacao = Publicacao::with('walledGardens')->find($publicacaoId);

        if (!$publicacao) {
            return response()->json([
                'success' => false,
                'message' => 'Publicação não encontrada.',
            ], 404);
        }

        $walledGardens = $publicacao->walledGardens()
            ->where('ativo', true)
            ->get()
            ->map(function ($wg) {
                return [
                    'tipo' => $wg->tipo,
                    'endereco' => $wg->endereco,
                    'descricao' => $wg->descricao,
                ];
            });

        return response()->json([
            'success' => true,
            'publicacao_id' => $publicacao->id,
            'count' => $walledGardens->count(),
            'data' => $walledGardens,
        ]);
    }

    /**
     * Registrar impressão de anúncio
     * 
     * POST /api/hotspot/impressao
     * Body: { "serial": "ABC123", "publicacao_id": 1, "criativo_id": 1, "usuarios_unicos": 5 }
     */
    public function registrarImpressao(Request $request)
    {
        $validated = $request->validate([
            'serial' => 'required|string|exists:hotspots,serial',
            'publicacao_id' => 'required|exists:publicacoes,id',
            'criativo_id' => 'required|exists:criativos,id',
            'usuarios_unicos' => 'nullable|integer|min:1',
        ]);

        $hotspot = Hotspot::where('serial', $validated['serial'])->first();
        $publicacao = Publicacao::find($validated['publicacao_id']);

        // Verificar se tracking está habilitado
        if (!$publicacao->track_impressoes) {
            return response()->json([
                'success' => false,
                'message' => 'Tracking de impressões desabilitado.',
            ], 400);
        }

        $data = now()->format('Y-m-d');

        // Buscar ou criar registro de impressão do dia
        $impressao = \App\Models\PublicacaoImpressao::firstOrNew([
            'hotspot_id' => $hotspot->id,
            'publicacao_id' => $validated['publicacao_id'],
            'criativo_id' => $validated['criativo_id'],
            'data' => $data,
        ]);

        $impressao->impressoes = ($impressao->impressoes ?? 0) + 1;
        $impressao->usuarios_unicos = max(
            $impressao->usuarios_unicos ?? 0,
            $validated['usuarios_unicos'] ?? 1
        );
        $impressao->save();

        return response()->json([
            'success' => true,
            'message' => 'Impressão registrada.',
            'data' => [
                'total_hoje' => $impressao->impressoes,
                'usuarios_unicos' => $impressao->usuarios_unicos,
            ]
        ]);
    }

    /**
     * Registrar clique em anúncio
     * 
     * POST /api/hotspot/clique
     * Body: { "serial": "ABC123", "publicacao_id": 1, "criativo_id": 1, "ip_usuario": "192.168.1.1" }
     */
    public function registrarClique(Request $request)
    {
        $validated = $request->validate([
            'serial' => 'required|string|exists:hotspots,serial',
            'publicacao_id' => 'required|exists:publicacoes,id',
            'criativo_id' => 'required|exists:criativos,id',
            'ip_usuario' => 'required|ip',
            'user_agent' => 'nullable|string',
        ]);

        $hotspot = Hotspot::where('serial', $validated['serial'])->first();
        $publicacao = Publicacao::find($validated['publicacao_id']);

        // Verificar se tracking está habilitado
        if (!$publicacao->track_cliques) {
            return response()->json([
                'success' => false,
                'message' => 'Tracking de cliques desabilitado.',
            ], 400);
        }

        // Verificar se link está habilitado
        if (!$publicacao->habilitar_clique || !$publicacao->link_destino) {
            return response()->json([
                'success' => false,
                'message' => 'Clique não habilitado para esta publicação.',
            ], 400);
        }

        // Registrar clique
        $clique = \App\Models\PublicacaoClique::create([
            'hotspot_id' => $hotspot->id,
            'publicacao_id' => $validated['publicacao_id'],
            'criativo_id' => $validated['criativo_id'],
            'ip_usuario' => $validated['ip_usuario'],
            'user_agent' => $validated['user_agent'] ?? $request->header('User-Agent'),
            'clicked_at' => now(),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Clique registrado.',
            'redirect_url' => $publicacao->link_destino,
            'data' => [
                'clique_id' => $clique->id,
                'timestamp' => $clique->clicked_at->toIso8601String(),
            ]
        ]);
    }
}
