<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Http\Requests\Auth\RegisterRequest;
use App\Mail\NovoCadastroPendente;
use App\Models\Empresa;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;

class RegisterController extends Controller
{
    /**
     * Exibe o formulário de registro
     */
    public function showForm()
    {
        return view('auth.register');
    }
    
    /**
     * Valida código de convite (AJAX)
     */
    public function validateCode(Request $request)
    {
        $request->validate([
            'codigo' => 'required|string|size:10',
        ]);
        
        $empresa = Empresa::where('codigo_convite', strtoupper($request->codigo))
                          ->whereIn('tipo_empresa', ['matriz', 'franquia'])
                          ->where('status', 0) // ativa
                          ->first();
        
        if (!$empresa) {
            return response()->json([
                'valid' => false,
                'message' => 'Código de convite inválido ou empresa desativada.'
            ], 422);
        }
        
        return response()->json([
            'valid' => true,
            'empresa' => [
                'id' => $empresa->id,
                'name' => $empresa->name,
                'email' => $empresa->email,
            ]
        ]);
    }
    
    /**
     * Processa o registro do novo usuário
     */
    public function register(RegisterRequest $request)
    {
        try {
            DB::beginTransaction();
            
            // Validar novamente o código (segurança)
            $empresa = Empresa::where('codigo_convite', strtoupper($request->codigo_convite))
                              ->whereIn('tipo_empresa', ['matriz', 'franquia'])
                              ->where('status', 0)
                              ->firstOrFail();
            
            // Criar usuário com status pendente
            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'password' => Hash::make($request->password),
                'empresa_id' => $empresa->id,
                'status_cadastro' => 'pendente',
            ]);
            
            DB::commit();
            
            // Verificar feature flags antes de notificar administradores
            if ($this->shouldSendEmail('novo_cadastro_pendente')) {
                // Buscar todos os administradores
                $admins = User::where('is_superadmin', true)
                              ->orWhere('status_cadastro', 'aprovado')
                              ->get();
                
                if (Config::get('features.dev_mode')) {
                    // Modo desenvolvimento - apenas loga
                    Log::info('Notificação de novo cadastro (dev mode)', [
                        'tipo' => 'NovoCadastroPendente',
                        'usuario' => $user->name,
                        'email' => $user->email,
                        'admins_notificados' => $admins->count(),
                    ]);
                } else {
                    // Enviar email real para cada administrador
                    foreach ($admins as $admin) {
                        $recipient = $this->getEmailRecipient($admin->email);
                        Mail::to($recipient)->queue(new NovoCadastroPendente($user));
                    }
                }
            }
            
            return redirect()->route('login')
                ->with('success', 'Cadastro realizado com sucesso! Aguarde a aprovação da sua conta.');
            
        } catch (\Exception $e) {
            DB::rollBack();
            
            return back()
                ->withInput()
                ->withErrors(['error' => 'Erro ao processar cadastro. Tente novamente.']);
        }
    }
    
    /**
     * Verifica se o envio de email está habilitado para um tipo específico
     */
    protected function shouldSendEmail(string $tipo): bool
    {
        // Verificar flag mestre primeiro
        if (!Config::get('features.email_enabled', true)) {
            return false;
        }
        
        // Verificar flag específica do tipo
        return Config::get("features.emails.{$tipo}", true);
    }
    
    /**
     * Retorna o destinatário do email (real ou de teste)
     */
    protected function getEmailRecipient(string $defaultEmail): string
    {
        // Se há email de teste configurado, redirecionar para ele
        $testRecipient = Config::get('features.test_recipient');
        
        if (!empty($testRecipient)) {
            Log::info('Email redirecionado para teste', [
                'original' => $defaultEmail,
                'teste' => $testRecipient,
            ]);
            
            return $testRecipient;
        }
        
        return $defaultEmail;
    }
}
