<?php

namespace App\Http\Controllers;

use App\Models\Empresa;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Intervention\Image\ImageManager;
use Intervention\Image\Drivers\Gd\Driver;

class EmpresaController extends Controller
{
    public function index(Request $request)
    {
        $query = Empresa::query();

        // Filtro por tipo de empresa (franquia, cliente, matriz)
        if ($request->filled('tipo')) {
            $query->where('tipo_empresa', $request->input('tipo'));
        }

        // Filtro por busca textual
        if ($request->filled('q')) {
            $query->where('name', 'like', '%' . $request->input('q') . '%');
        }

        // Ordenação: Matriz -> Franquia -> Cliente (alfabético dentro de cada grupo)
        $empresas = $query
            ->orderByRaw("FIELD(tipo_empresa, 'matriz', 'franquia', 'cliente')")
            ->orderBy('name', 'asc')
            ->paginate(15);

        return view('empresas.index', compact('empresas'));
    }

    public function create()
    {
        return view('empresas.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'tipo_empresa' => 'required|in:matriz,franquia,cliente',
            'empresa_pai_id' => 'nullable|exists:empresas,id',
            'email' => 'nullable|email|max:255|unique:empresas',
            'phone' => 'nullable|string|max:255',
            'logomarca' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $data = $request->all();

        // Upload e redimensionamento da logomarca
        if ($request->hasFile('logomarca')) {
            $file = $request->file('logomarca');
            $filename = time() . '_' . $file->getClientOriginalName();
            
            // Redimensionar a imagem para máximo 350px de largura
            $manager = new ImageManager(new Driver());
            $image = $manager->read($file);
            
            // Redimensionar mantendo proporção, máximo 350px de largura
            $image->scaleDown(350);
            
            // Salvar a imagem redimensionada
            $path = 'logos/' . $filename;
            Storage::disk('public')->put($path, $image->encode());
            
            $data['logomarca'] = $path;
        }

        Empresa::create($data);

        return redirect()->route('empresas.index');
    }

    public function edit(Empresa $empresa)
    {
        return view('empresas.edit', compact('empresa'));
    }

    public function update(Request $request, Empresa $empresa)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'tipo_empresa' => 'required|in:matriz,franquia,cliente',
            'empresa_pai_id' => 'nullable|exists:empresas,id',
            'email' => 'nullable|email|max:255|unique:empresas,email,'.$empresa->id,
            'phone' => 'nullable|string|max:255',
            'logomarca' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $data = $request->all();

        // Upload e redimensionamento da logomarca
        if ($request->hasFile('logomarca')) {
            // Deletar logomarca anterior se existir
            if ($empresa->logomarca && Storage::disk('public')->exists($empresa->logomarca)) {
                Storage::disk('public')->delete($empresa->logomarca);
            }
            
            $file = $request->file('logomarca');
            $filename = time() . '_' . $file->getClientOriginalName();
            
            // Redimensionar a imagem para máximo 350px de largura
            $manager = new ImageManager(new Driver());
            $image = $manager->read($file);
            
            // Redimensionar mantendo proporção, máximo 350px de largura
            $image->scaleDown(350);
            
            // Salvar a imagem redimensionada
            $path = 'logos/' . $filename;
            Storage::disk('public')->put($path, $image->encode());
            
            $data['logomarca'] = $path;
        }

        $empresa->update($data);

        return redirect()->route('empresas.index');
    }

    public function destroy(Empresa $empresa)
    {
        // Deletar logomarca se existir
        if ($empresa->logomarca && Storage::disk('public')->exists($empresa->logomarca)) {
            Storage::disk('public')->delete($empresa->logomarca);
        }
        
        $empresa->delete();

        return redirect()->route('empresas.index');
    }

    public function toggleStatus(Empresa $empresa)
    {
        $empresa->status = !$empresa->status;
        $empresa->save();

        return redirect()->route('empresas.index');
    }

    public function trash()
    {
        $empresas = Empresa::onlyTrashed()
                        ->with('destroyer:id,name')
                        ->latest('deleted_at')
                        ->get();
        return response()->json($empresas);
    }

    public function restore($id)
    {
        $empresa = Empresa::onlyTrashed()->findOrFail($id);
        $empresa->restore();

        return redirect()->route('empresas.index'); // Redirect to index as there is no trash page
    }
}
