<?php

namespace App\Http\Controllers;

use App\Models\Publicacao;
use App\Models\PublicacaoImpressao;
use App\Models\PublicacaoClique;
use App\Models\Hotspot;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class EstatisticasController extends Controller
{
    /**
     * Dashboard de estatísticas gerais
     */
    public function dashboard(Request $request)
    {
        $dataInicio = $request->input('data_inicio', now()->subDays(30)->format('Y-m-d'));
        $dataFim = $request->input('data_fim', now()->format('Y-m-d'));

        // Estatísticas gerais
        $stats = [
            'total_impressoes' => PublicacaoImpressao::whereBetween('data', [$dataInicio, $dataFim])
                ->sum('impressoes'),
            'total_usuarios_unicos' => PublicacaoImpressao::whereBetween('data', [$dataInicio, $dataFim])
                ->sum('usuarios_unicos'),
            'total_cliques' => PublicacaoClique::whereBetween('clicked_at', [$dataInicio, $dataFim])
                ->count(),
            'ctr_geral' => 0,
            'publicacoes_ativas' => Publicacao::ativas()->count(),
            'hotspots_ativos' => Hotspot::where('status', true)->count(),
        ];

        // Calcular CTR geral
        if ($stats['total_impressoes'] > 0) {
            $stats['ctr_geral'] = round(($stats['total_cliques'] / $stats['total_impressoes']) * 100, 2);
        }

        // Top 10 publicações por impressões
        $topPublicacoes = PublicacaoImpressao::with('publicacao')
            ->whereBetween('data', [$dataInicio, $dataFim])
            ->select('publicacao_id', DB::raw('SUM(impressoes) as total_impressoes'))
            ->groupBy('publicacao_id')
            ->orderByDesc('total_impressoes')
            ->limit(10)
            ->get();

        // Top 10 hotspots por impressões
        $topHotspots = PublicacaoImpressao::with('hotspot')
            ->whereBetween('data', [$dataInicio, $dataFim])
            ->select('hotspot_id', DB::raw('SUM(impressoes) as total_impressoes'))
            ->groupBy('hotspot_id')
            ->orderByDesc('total_impressoes')
            ->limit(10)
            ->get();

        // Impressões por dia
        $impressoesPorDia = PublicacaoImpressao::whereBetween('data', [$dataInicio, $dataFim])
            ->select('data', DB::raw('SUM(impressoes) as total'))
            ->groupBy('data')
            ->orderBy('data')
            ->get();

        // Cliques por dia
        $cliquesPorDia = PublicacaoClique::whereBetween('clicked_at', [$dataInicio, $dataFim])
            ->select(DB::raw('DATE(clicked_at) as data'), DB::raw('COUNT(*) as total'))
            ->groupBy('data')
            ->orderBy('data')
            ->get();

        return view('estatisticas.dashboard', compact(
            'stats',
            'topPublicacoes',
            'topHotspots',
            'impressoesPorDia',
            'cliquesPorDia',
            'dataInicio',
            'dataFim'
        ));
    }

    /**
     * Relatório de performance por publicação
     */
    public function porPublicacao(Request $request)
    {
        $dataInicio = $request->input('data_inicio', now()->subDays(30)->format('Y-m-d'));
        $dataFim = $request->input('data_fim', now()->format('Y-m-d'));

        $publicacoes = Publicacao::with(['criativo', 'franquia', 'cliente'])
            ->withCount([
                'impressoes as total_impressoes' => function ($query) use ($dataInicio, $dataFim) {
                    $query->whereBetween('data', [$dataInicio, $dataFim])
                        ->select(DB::raw('SUM(impressoes)'));
                },
                'cliques as total_cliques' => function ($query) use ($dataInicio, $dataFim) {
                    $query->whereBetween('clicked_at', [$dataInicio, $dataFim]);
                }
            ])
            ->get()
            ->map(function ($pub) {
                $pub->ctr = $pub->total_impressoes > 0
                    ? round(($pub->total_cliques / $pub->total_impressoes) * 100, 2)
                    : 0;
                return $pub;
            })
            ->sortByDesc('total_impressoes');

        return view('estatisticas.por-publicacao', compact('publicacoes', 'dataInicio', 'dataFim'));
    }

    /**
     * Relatório de performance por hotspot
     */
    public function porHotspot(Request $request)
    {
        $dataInicio = $request->input('data_inicio', now()->subDays(30)->format('Y-m-d'));
        $dataFim = $request->input('data_fim', now()->format('Y-m-d'));

        $hotspots = Hotspot::with(['franquia', 'cliente'])
            ->withCount([
                'impressoes as total_impressoes' => function ($query) use ($dataInicio, $dataFim) {
                    $query->whereBetween('data', [$dataInicio, $dataFim])
                        ->select(DB::raw('SUM(impressoes)'));
                },
                'cliques as total_cliques' => function ($query) use ($dataInicio, $dataFim) {
                    $query->whereBetween('clicked_at', [$dataInicio, $dataFim]);
                }
            ])
            ->get()
            ->map(function ($hotspot) {
                $hotspot->ctr = $hotspot->total_impressoes > 0
                    ? round(($hotspot->total_cliques / $hotspot->total_impressoes) * 100, 2)
                    : 0;
                return $hotspot;
            })
            ->sortByDesc('total_impressoes');

        return view('estatisticas.por-hotspot', compact('hotspots', 'dataInicio', 'dataFim'));
    }

    /**
     * API: Dados para gráficos (JSON)
     */
    public function graficosJson(Request $request)
    {
        $tipo = $request->input('tipo', 'impressoes');
        $dataInicio = $request->input('data_inicio', now()->subDays(30)->format('Y-m-d'));
        $dataFim = $request->input('data_fim', now()->format('Y-m-d'));

        switch ($tipo) {
            case 'impressoes_diarias':
                $data = PublicacaoImpressao::whereBetween('data', [$dataInicio, $dataFim])
                    ->select('data', DB::raw('SUM(impressoes) as total'))
                    ->groupBy('data')
                    ->orderBy('data')
                    ->get();
                break;

            case 'cliques_diarios':
                $data = PublicacaoClique::whereBetween('clicked_at', [$dataInicio, $dataFim])
                    ->select(DB::raw('DATE(clicked_at) as data'), DB::raw('COUNT(*) as total'))
                    ->groupBy('data')
                    ->orderBy('data')
                    ->get();
                break;

            case 'cliques_por_hora':
                $data = PublicacaoClique::whereBetween('clicked_at', [$dataInicio, $dataFim])
                    ->select(DB::raw('HOUR(clicked_at) as hora'), DB::raw('COUNT(*) as total'))
                    ->groupBy('hora')
                    ->orderBy('hora')
                    ->get();
                break;

            case 'ctr_por_publicacao':
                $data = Publicacao::with('criativo')
                    ->select('publicacoes.id', 'publicacoes.titulo')
                    ->leftJoin('publicacao_impressoes', 'publicacoes.id', '=', 'publicacao_impressoes.publicacao_id')
                    ->leftJoin('publicacao_cliques', 'publicacoes.id', '=', 'publicacao_cliques.publicacao_id')
                    ->whereBetween('publicacao_impressoes.data', [$dataInicio, $dataFim])
                    ->groupBy('publicacoes.id', 'publicacoes.titulo')
                    ->select(
                        'publicacoes.id',
                        'publicacoes.titulo',
                        DB::raw('SUM(publicacao_impressoes.impressoes) as total_impressoes'),
                        DB::raw('COUNT(DISTINCT publicacao_cliques.id) as total_cliques'),
                        DB::raw('ROUND((COUNT(DISTINCT publicacao_cliques.id) / SUM(publicacao_impressoes.impressoes)) * 100, 2) as ctr')
                    )
                    ->having('total_impressoes', '>', 0)
                    ->orderByDesc('ctr')
                    ->limit(10)
                    ->get();
                break;

            default:
                return response()->json(['error' => 'Tipo de gráfico inválido'], 400);
        }

        return response()->json($data);
    }
}
