<?php

namespace App\Models;

use App\Traits\Userstamps;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Empresa extends Model
{
    use HasFactory, SoftDeletes, Userstamps;

    protected $fillable = [
        'name', 
        'email', 
        'phone', 
        'logomarca', 
        'status',
        'tipo_empresa',
        'empresa_pai_id',
        'codigo_convite'
    ];

    protected $casts = [
        'tipo_empresa' => 'string',
    ];

    // ==========================================
    // RELACIONAMENTOS HIERARQUIA
    // ==========================================
    
    /**
     * Empresa pai (matriz ou franquia)
     */
    public function empresaPai()
    {
        return $this->belongsTo(Empresa::class, 'empresa_pai_id');
    }
    
    /**
     * Franquias (se for matriz)
     */
    public function franquias()
    {
        return $this->hasMany(Empresa::class, 'empresa_pai_id')
                    ->where('tipo_empresa', 'franquia');
    }
    
    /**
     * Clientes (se for matriz ou franquia)
     */
    public function clientes()
    {
        return $this->hasMany(Empresa::class, 'empresa_pai_id')
                    ->where('tipo_empresa', 'cliente');
    }

    // ==========================================
    // RELACIONAMENTOS LEGADOS
    // ==========================================
    
    public function midias()
    {
        return $this->hasMany(Midia::class);
    }

    /**
     * Hotspots como CLIENTE
     */
    public function hotspots()
    {
        return $this->hasMany(Hotspot::class, 'cliente_id');
    }
    
    /**
     * Hotspots gerenciados (se for franquia)
     */
    public function hotspotsGerenciados()
    {
        return $this->hasMany(Hotspot::class, 'franquia_id');
    }
    
    /**
     * Usuários da empresa
     */
    public function usuarios()
    {
        return $this->hasMany(User::class, 'empresa_id');
    }
    
    /**
     * Alias em inglês para usuarios()
     * Mantém consistência com o Model User
     */
    public function users()
    {
        return $this->usuarios();
    }
    
    /**
     * Publicações da empresa (se for cliente)
     */
    public function publicacoes()
    {
        return $this->hasMany(Publicacao::class, 'cliente_id');
    }
    
    /**
     * Formatos permitidos (se for franquia)
     */
    public function formatosPermitidos()
    {
        return $this->hasMany(FranquiaFormatoPermitido::class, 'franquia_id');
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function updater()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    public function destroyer()
    {
        return $this->belongsTo(User::class, 'deleted_by');
    }
}
