<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class PublicacaoClique extends Model
{
    /** @use HasFactory<\Database\Factories\PublicacaoCliqueFactory> */
    use HasFactory;

    /**
     * Nome da tabela
     */
    protected $table = 'publicacao_cliques';

    /**
     * Campos que podem ser preenchidos em massa
     */
    protected $fillable = [
        'hotspot_id',
        'publicacao_id',
        'criativo_id',
        'ip_usuario',
        'user_agent',
        'clicked_at',
    ];

    /**
     * Casting de atributos
     */
    protected $casts = [
        'clicked_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Relacionamento: Clique pertence a um Hotspot
     */
    public function hotspot(): BelongsTo
    {
        return $this->belongsTo(Hotspot::class);
    }

    /**
     * Relacionamento: Clique pertence a uma Publicação
     */
    public function publicacao(): BelongsTo
    {
        return $this->belongsTo(Publicacao::class);
    }

    /**
     * Relacionamento: Clique pertence a um Criativo (antiga Mídia)
     */
    public function criativo(): BelongsTo
    {
        return $this->belongsTo(Criativo::class);
    }

    /**
     * Scopes para consultas comuns
     */
    public function scopePorData($query, $dataInicio, $dataFim = null)
    {
        if ($dataFim) {
            return $query->whereBetween('clicked_at', [$dataInicio, $dataFim]);
        }
        return $query->whereDate('clicked_at', $dataInicio);
    }

    public function scopePorPublicacao($query, int $publicacaoId)
    {
        return $query->where('publicacao_id', $publicacaoId);
    }

    public function scopePorHotspot($query, int $hotspotId)
    {
        return $query->where('hotspot_id', $hotspotId);
    }

    public function scopeHoje($query)
    {
        return $query->whereDate('clicked_at', now()->toDateString());
    }

    /**
     * Agrupa cliques por hora para análise de pico
     */
    public function scopeAgrupadoPorHora($query)
    {
        return $query->selectRaw('HOUR(clicked_at) as hora, COUNT(*) as total')
            ->groupBy('hora')
            ->orderBy('hora');
    }
}
