<?php

namespace Database\Factories;

use App\Models\HotspotCustomization;
use App\Models\Hotspot;
use App\Models\User;
use Illuminate\Database\Eloquent\Factories\Factory;

/**
 * @extends \Illuminate\Database\Eloquent\Factories\Factory<\App\Models\HotspotCustomization>
 */
class HotspotCustomizationFactory extends Factory
{
    protected $model = HotspotCustomization::class;

    /**
     * Define the model's default state.
     *
     * @return array<string, mixed>
     */
    public function definition(): array
    {
        return [
            'hotspot_id' => Hotspot::factory(),

            // Identidade Visual
            'brand_name' => fake()->company(),
            'logo_url' => 'img/logo-' . fake()->uuid() . '.png',
            'favicon_url' => 'img/favicon-' . fake()->uuid() . '.ico',

            // Cores - Background Gradient
            'bg_color_1' => 'hsla(' . fake()->numberBetween(0, 360) . ', ' . fake()->numberBetween(0, 100) . '%, ' . fake()->numberBetween(20, 80) . '%, 1)',
            'bg_color_2' => 'hsla(' . fake()->numberBetween(0, 360) . ', ' . fake()->numberBetween(0, 100) . '%, ' . fake()->numberBetween(20, 80) . '%, 1)',
            'bg_color_3' => 'hsla(' . fake()->numberBetween(0, 360) . ', ' . fake()->numberBetween(0, 100) . '%, ' . fake()->numberBetween(20, 80) . '%, 1)',
            'bg_color_4' => 'hsla(' . fake()->numberBetween(0, 360) . ', ' . fake()->numberBetween(0, 100) . '%, ' . fake()->numberBetween(20, 80) . '%, 1)',

            // Cores - Elementos
            'primary_color' => fake()->hexColor(),
            'primary_hover_color' => fake()->hexColor(),
            'text_color' => fake()->hexColor(),
            'link_color' => fake()->hexColor(),

            // Textos
            'welcome_title' => fake()->sentence(3),
            'login_title' => fake()->company(),
            'register_title' => 'Cadastro - ' . fake()->word(),
            'status_title' => fake()->sentence(2),

            // Botões
            'btn_first_access' => 'Primeiro Acesso',
            'btn_login' => 'Entrar',
            'btn_connect' => 'Conectar',
            'btn_disconnect' => 'Sair',

            // Placeholders
            'placeholder_username' => 'Digite seu usuário',
            'placeholder_password' => 'Digite sua senha',
            'placeholder_name' => 'Seu nome completo',
            'placeholder_email' => 'seu@email.com',
            'placeholder_phone' => '(11) 99999-9999',

            // Tempos
            'modal_countdown_seconds' => fake()->numberBetween(5, 30),
            'banner_rotation_seconds' => fake()->numberBetween(3, 10),
            'status_refresh_seconds' => fake()->numberBetween(30, 120),

            // Propaganda
            'enable_advertisement' => fake()->boolean(80), // 80% chance de true

            // URLs
            'terms_url' => fake()->boolean(50) ? fake()->url() : null,
            'privacy_url' => fake()->boolean(50) ? fake()->url() : null,
            'support_url' => fake()->boolean(50) ? fake()->url() : null,
            'redirect_after_login' => fake()->url(),

            // Campos do Formulário
            'require_name' => fake()->boolean(90),
            'require_email' => fake()->boolean(80),
            'require_phone' => fake()->boolean(70),
            'require_birth_date' => fake()->boolean(30),
            'require_address' => fake()->boolean(20),

            // Tema
            'theme_preset' => fake()->randomElement(['default', 'lite', 'dark', 'custom']),

            // Configurações Técnicas
            'enable_cpf_mask' => fake()->boolean(90),
            'enable_phone_mask' => fake()->boolean(90),
            'enable_password_hash' => fake()->boolean(95),

            // Informações Exibidas
            'show_hotspot_identity' => fake()->boolean(80),
            'show_mac_address' => fake()->boolean(70),
            'show_ip_address' => fake()->boolean(70),
            'show_upload_download' => fake()->boolean(85),
            'show_session_time' => fake()->boolean(90),

            // CSS/JS Customizado
            'custom_css' => fake()->boolean(30) ? '/* CSS customizado */\n.custom { color: red; }' : null,
            'custom_js' => fake()->boolean(20) ? '// JS customizado\nconsole.log("Custom JS loaded");' : null,
            'custom_head_html' => fake()->boolean(15) ? '<meta name="custom" content="test">' : null,
            'custom_body_html' => fake()->boolean(15) ? '<div class="custom">Test</div>' : null,

            // Controle
            'is_active' => fake()->boolean(90),
            'created_by' => User::factory(),
            'updated_by' => User::factory(),
        ];
    }

    /**
     * Customização ativa
     */
    public function active(): static
    {
        return $this->state(fn (array $attributes) => [
            'is_active' => true,
        ]);
    }

    /**
     * Customização inativa
     */
    public function inactive(): static
    {
        return $this->state(fn (array $attributes) => [
            'is_active' => false,
        ]);
    }

    /**
     * Customização com tema específico
     */
    public function theme(string $theme): static
    {
        return $this->state(fn (array $attributes) => [
            'theme_preset' => $theme,
        ]);
    }

    /**
     * Customização com CSS/JS customizado
     */
    public function withCustomCode(): static
    {
        return $this->state(fn (array $attributes) => [
            'custom_css' => "/* Custom CSS for testing */\n.test { background: blue; }",
            'custom_js' => "// Custom JS for testing\nconsole.log('Test loaded');",
            'custom_head_html' => '<meta name="test" content="testing">',
            'custom_body_html' => '<div id="test-div">Testing</div>',
        ]);
    }

    /**
     * Customização minimalista (sem customizações extras)
     */
    public function minimal(): static
    {
        return $this->state(fn (array $attributes) => [
            'custom_css' => null,
            'custom_js' => null,
            'custom_head_html' => null,
            'custom_body_html' => null,
            'terms_url' => null,
            'privacy_url' => null,
            'support_url' => null,
        ]);
    }
}
