<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Renomear tabela midias para criativos
        Schema::rename('midias', 'criativos');
        
        // Adicionar novos campos
        Schema::table('criativos', function (Blueprint $table) {
            // Franquia dona do criativo (null = FreeFi/catálogo)
            $table->foreignId('franquia_id')
                  ->nullable()
                  ->after('id')
                  ->constrained('empresas')
                  ->onDelete('restrict');
            
            // Renomear campo 'type' para 'tipo' e adicionar novos valores
            // Obs: campo 'type' atual será dropado e recriado como 'tipo'
            
            // Formato do arquivo (baseado no type atual)
            $table->enum('formato', ['jpg', 'png', 'gif', 'mp4', 'html'])
                  ->default('jpg')
                  ->after('title');
            
            // Conteúdo HTML (para quiz/NPS)
            $table->text('conteudo_html')
                  ->nullable()
                  ->after('path');
            
            // Status de aprovação
            $table->enum('status_aprovacao', ['pendente', 'aprovado', 'rejeitado', 'bloqueado'])
                  ->default('aprovado') // criativos existentes aprovados automaticamente
                  ->after('conteudo_html');
            
            // Quem aprovou
            $table->foreignId('aprovado_por')
                  ->nullable()
                  ->after('status_aprovacao')
                  ->constrained('users')
                  ->onDelete('set null');
            
            // Quando foi aprovado
            $table->timestamp('aprovado_em')
                  ->nullable()
                  ->after('aprovado_por');
            
            // Motivo da rejeição
            $table->text('motivo_rejeicao')
                  ->nullable()
                  ->after('aprovado_em');
            
            // Bloqueio permanente (só SuperAdmin)
            $table->boolean('bloqueado_permanente')
                  ->default(false)
                  ->after('motivo_rejeicao');
            
            // Índices
            $table->index('franquia_id');
            $table->index('formato');
            $table->index('status_aprovacao');
        });
        
        // Renomear campo 'type' para 'tipo' e expandir enum
        Schema::table('criativos', function (Blueprint $table) {
            // Dropar campo antigo 'type'
            $table->dropColumn('type');
            
            // Criar novo campo 'tipo' com mais opções
            $table->enum('tipo', ['imagem', 'video', 'html', 'quiz', 'nps'])
                  ->default('imagem')
                  ->after('franquia_id');
        });
        
        // Renomear campo empresa_id para manter compatibilidade temporária
        // (será usado como referência histórica)
        Schema::table('criativos', function (Blueprint $table) {
            $table->renameColumn('empresa_id', 'empresa_legado_id');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // Reverter nome da coluna
        Schema::table('criativos', function (Blueprint $table) {
            $table->renameColumn('empresa_legado_id', 'empresa_id');
        });
        
        Schema::table('criativos', function (Blueprint $table) {
            // Recriar campo type original
            $table->string('type')->after('title');
            
            // Dropar campo tipo
            $table->dropColumn('tipo');
            
            // Dropar novos campos e índices
            $table->dropForeign(['franquia_id']);
            $table->dropForeign(['aprovado_por']);
            $table->dropIndex(['franquia_id']);
            $table->dropIndex(['formato']);
            $table->dropIndex(['status_aprovacao']);
            $table->dropColumn([
                'franquia_id',
                'formato',
                'conteudo_html',
                'status_aprovacao',
                'aprovado_por',
                'aprovado_em',
                'motivo_rejeicao',
                'bloqueado_permanente'
            ]);
        });
        
        // Renomear de volta para midias
        Schema::rename('criativos', 'midias');
    }
};
