<?php

namespace Database\Seeders;

use App\Models\Empresa;
use App\Models\User;
use Illuminate\Database\Seeder;
use Illuminate\Support\Str;

class EmpresaHierarquiaSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Criar usuário SuperAdmin para ser o created_by
        $superAdmin = User::firstOrCreate(
            ['email' => 'admin@freefi.com.br'],
            [
                'name' => 'Super Admin FreeFi',
                'password' => bcrypt('password'),
                'status_cadastro' => 'aprovado',
                'aprovado_em' => now(),
            ]
        );

        // ==========================================
        // 1. CRIAR MATRIZ (FreeFi)
        // ==========================================
        $freefi = Empresa::create([
            'name' => 'FreeFi - Matriz',
            'tipo_empresa' => 'matriz',
            'empresa_pai_id' => null,
            'codigo_convite' => null, // Matriz não tem código
            'status' => 0, // Ativo
            'created_by' => $superAdmin->id,
        ]);

        echo "✅ Matriz criada: {$freefi->name}\n";

        // Associar SuperAdmin à matriz
        $superAdmin->update(['empresa_id' => $freefi->id]);

        // ==========================================
        // 2. CRIAR FRANQUIAS com Códigos de Convite
        // ==========================================
        $franquias = [];

        // Franquia 1: São Paulo
        $franquias['sp'] = Empresa::create([
            'name' => 'FreeFi São Paulo',
            'tipo_empresa' => 'franquia',
            'empresa_pai_id' => $freefi->id,
            'codigo_convite' => $this->gerarCodigoConvite(),
            'status' => 0,
            'created_by' => $superAdmin->id,
        ]);

        // Franquia 2: Rio de Janeiro
        $franquias['rj'] = Empresa::create([
            'name' => 'FreeFi Rio de Janeiro',
            'tipo_empresa' => 'franquia',
            'empresa_pai_id' => $freefi->id,
            'codigo_convite' => $this->gerarCodigoConvite(),
            'status' => 0,
            'created_by' => $superAdmin->id,
        ]);

        // Franquia 3: Minas Gerais
        $franquias['mg'] = Empresa::create([
            'name' => 'FreeFi Minas Gerais',
            'tipo_empresa' => 'franquia',
            'empresa_pai_id' => $freefi->id,
            'codigo_convite' => $this->gerarCodigoConvite(),
            'status' => 0,
            'created_by' => $superAdmin->id,
        ]);

        // Franquia 4: Bahia (Desativada para teste)
        $franquias['ba'] = Empresa::create([
            'name' => 'FreeFi Bahia',
            'tipo_empresa' => 'franquia',
            'empresa_pai_id' => $freefi->id,
            'codigo_convite' => $this->gerarCodigoConvite(),
            'status' => 1, // Desativada
            'created_by' => $superAdmin->id,
        ]);

        foreach ($franquias as $uf => $franquia) {
            echo "✅ Franquia criada: {$franquia->name} - Código: {$franquia->codigo_convite}\n";
        }

        // ==========================================
        // 3. CRIAR CLIENTES (3 por franquia)
        // ==========================================
        
        // Clientes SP
        $clientesSP = [
            'Shopping Vila Olímpia' => 0,
            'Restaurante Fogo de Chão' => 0,
            'Hotel Ibis Paulista' => 1, // Desativado
        ];

        foreach ($clientesSP as $nome => $status) {
            Empresa::create([
                'name' => $nome,
                'tipo_empresa' => 'cliente',
                'empresa_pai_id' => $franquias['sp']->id,
                'codigo_convite' => null, // Cliente não tem código
                'status' => $status,
                'created_by' => $superAdmin->id,
            ]);
            echo "  └─ Cliente SP: {$nome}\n";
        }

        // Clientes RJ
        $clientesRJ = [
            'Shopping Rio Sul' => 0,
            'Bar Urca Point' => 0,
            'Café Colombo Centro' => 0,
        ];

        foreach ($clientesRJ as $nome => $status) {
            Empresa::create([
                'name' => $nome,
                'tipo_empresa' => 'cliente',
                'empresa_pai_id' => $franquias['rj']->id,
                'codigo_convite' => null,
                'status' => $status,
                'created_by' => $superAdmin->id,
            ]);
            echo "  └─ Cliente RJ: {$nome}\n";
        }

        // Clientes MG
        $clientesMG = [
            'Shopping BH' => 0,
            'Padaria Pão de Queijo Mineiro' => 0,
        ];

        foreach ($clientesMG as $nome => $status) {
            Empresa::create([
                'name' => $nome,
                'tipo_empresa' => 'cliente',
                'empresa_pai_id' => $franquias['mg']->id,
                'codigo_convite' => null,
                'status' => $status,
                'created_by' => $superAdmin->id,
            ]);
            echo "  └─ Cliente MG: {$nome}\n";
        }

        // Clientes BA (franquia desativada)
        Empresa::create([
            'name' => 'Hotel Salvador Mar',
            'tipo_empresa' => 'cliente',
            'empresa_pai_id' => $franquias['ba']->id,
            'codigo_convite' => null,
            'status' => 0,
            'created_by' => $superAdmin->id,
        ]);
        echo "  └─ Cliente BA: Hotel Salvador Mar\n";

        echo "\n";
        echo "📊 RESUMO:\n";
        echo "   • 1 Matriz (FreeFi)\n";
        echo "   • 4 Franquias (3 ativas, 1 desativada)\n";
        echo "   • 9 Clientes (8 ativos, 1 desativado)\n";
        echo "   • Total: " . Empresa::count() . " empresas\n";
        echo "\n";
        echo "🔑 CÓDIGOS DE CONVITE:\n";
        foreach ($franquias as $uf => $franquia) {
            echo "   • {$franquia->name}: {$franquia->codigo_convite}\n";
        }
    }

    /**
     * Gera código de convite único (4 letras + 6 dígitos)
     */
    private function gerarCodigoConvite(): string
    {
        do {
            // 4 letras maiúsculas
            $letras = strtoupper(Str::random(4));
            
            // 6 dígitos
            $numeros = str_pad(rand(0, 999999), 6, '0', STR_PAD_LEFT);
            
            $codigo = $letras . $numeros;
            
            // Verificar se já existe
            $existe = Empresa::where('codigo_convite', $codigo)->exists();
            
        } while ($existe);

        return $codigo;
    }
}
