<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\LogAcessoWifi;
use App\Models\Hotspot;
use App\Models\Empresa;
use Carbon\Carbon;

class LogAcessoWifiSeeder extends Seeder
{
    /**
     * Seed logs de acesso para testes
     */
    public function run(): void
    {
        $publicacaoId = 4; // ID da publicação
        $quantidade = 86; // Quantidade de logs a criar

        // Buscar hotspots disponíveis
        $hotspots = Hotspot::take(5)->get();
        if ($hotspots->isEmpty()) {
            $this->command->error('Nenhum hotspot encontrado. Crie hotspots primeiro.');
            return;
        }

        // Buscar empresas clientes
        $clientes = Empresa::where('tipo_empresa', 'cliente')->take(3)->get();
        if ($clientes->isEmpty()) {
            $this->command->warn('Nenhum cliente encontrado. Usando NULL para cliente_id.');
        }

        // Buscar franquia
        $franquia = Empresa::where('tipo_empresa', 'franquia')->first();

        $dispositivos = ['mobile', 'desktop', 'tablet'];
        $userAgents = [
            'Mozilla/5.0 (Linux; Android 13) AppleWebKit/537.36',
            'Mozilla/5.0 (Windows NT 10.0; Win64; x64) Chrome/120.0',
            'Mozilla/5.0 (iPhone; CPU iPhone OS 17_0) Safari/604.1',
            'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) Safari/605.1',
            'Mozilla/5.0 (Linux; Android 12; SM-G998B) Chrome/119.0',
        ];

        $this->command->info("Criando {$quantidade} logs de acesso para publicação ID {$publicacaoId}...");

        // Distribuir logs pelos últimos 7 dias
        $dataInicio = Carbon::now()->subDays(7);
        $dataFim = Carbon::now();

        for ($i = 0; $i < $quantidade; $i++) {
            // Gerar data aleatória entre início e fim
            $dataAcesso = Carbon::createFromTimestamp(
                rand($dataInicio->timestamp, $dataFim->timestamp)
            );

            // Gerar MAC address hash único
            $macHash = hash('sha256', 'mac_' . $i . '_' . $dataAcesso->timestamp);

            // IP aleatório (privado 192.168.x.x)
            $ip = '192.168.' . rand(1, 255) . '.' . rand(1, 254);

            // Hotspot aleatório
            $hotspot = $hotspots->random();

            // Cliente aleatório (pode ser null)
            $cliente = $clientes->isNotEmpty() ? $clientes->random() : null;

            // Tempo de sessão (entre 1 min e 30 min)
            $tempoSessao = rand(60, 1800);

            LogAcessoWifi::create([
                'hotspot_id' => $hotspot->id,
                'franquia_id' => $franquia?->id,
                'cliente_id' => $cliente?->id,
                'mac_address_hash' => $macHash,
                'ip_address' => $ip,
                'user_agent' => $userAgents[array_rand($userAgents)],
                'publicacao_visualizada_id' => $publicacaoId,
                'tempo_sessao' => $tempoSessao,
                'dispositivo_tipo' => $dispositivos[array_rand($dispositivos)],
                'created_at' => $dataAcesso,
            ]);

            if (($i + 1) % 10 == 0) {
                $this->command->info("Criados " . ($i + 1) . "/{$quantidade} logs...");
            }
        }

        $this->command->info("✅ {$quantidade} logs criados com sucesso!");
        $this->command->info("   Usuários únicos: " . LogAcessoWifi::where('publicacao_visualizada_id', $publicacaoId)
            ->distinct('mac_address_hash')->count('mac_address_hash'));
    }
}
