#!/bin/bash

# =============================================================================
# SCRIPT DE DEPLOY - CORREÇÃO DE PAGINAÇÃO
# FreeFi Admin - Bootstrap 5 Pagination Fix
# =============================================================================

set -e  # Exit on error

# Cores para output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Configurações
SERVER_IP="172.233.21.57"
SERVER_PORT="8082"
APP_PATH="/var/www/html/laravel/admin/freefi_admin"  # Ajustar conforme necessário

echo -e "${BLUE}========================================${NC}"
echo -e "${BLUE}  DEPLOY - CORREÇÃO DE PAGINAÇÃO${NC}"
echo -e "${BLUE}========================================${NC}"
echo ""

# Função para logging
log_info() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

log_success() {
    echo -e "${GREEN}[OK]${NC} $1"
}

log_warning() {
    echo -e "${YELLOW}[AVISO]${NC} $1"
}

log_error() {
    echo -e "${RED}[ERRO]${NC} $1"
}

# Verificar se está no diretório correto
if [ ! -f "artisan" ]; then
    log_error "Arquivo artisan não encontrado. Execute este script no diretório raiz do Laravel."
    exit 1
fi

log_info "Diretório validado: $(pwd)"

# 1. BACKUP
log_info "Criando backup dos arquivos..."
BACKUP_DIR="backups/paginacao-$(date +%Y%m%d_%H%M%S)"
mkdir -p "$BACKUP_DIR"

cp app/Providers/AppServiceProvider.php "$BACKUP_DIR/" 2>/dev/null || true
cp resources/views/publicacoes/index.blade.php "$BACKUP_DIR/" 2>/dev/null || true

log_success "Backup criado em: $BACKUP_DIR"

# 2. VERIFICAR SE ARQUIVOS EXISTEM
log_info "Verificando arquivos..."

if [ ! -f "app/Providers/AppServiceProvider.php" ]; then
    log_error "AppServiceProvider.php não encontrado!"
    exit 1
fi

if [ ! -f "resources/views/publicacoes/index.blade.php" ]; then
    log_error "publicacoes/index.blade.php não encontrado!"
    exit 1
fi

log_success "Arquivos encontrados"

# 3. VERIFICAR SE CORREÇÃO JÁ FOI APLICADA
log_info "Verificando se correção já foi aplicada..."

if grep -q "Paginator::useBootstrapFive()" app/Providers/AppServiceProvider.php; then
    log_success "✅ AppServiceProvider já contém Paginator::useBootstrapFive()"
else
    log_warning "⚠️ AppServiceProvider PRECISA ser atualizado manualmente!"
    echo ""
    echo "Adicione as seguintes linhas em app/Providers/AppServiceProvider.php:"
    echo ""
    echo "use Illuminate\Pagination\Paginator;  // No topo do arquivo"
    echo ""
    echo "// No método boot():"
    echo "Paginator::useBootstrapFive();"
    echo ""
    read -p "Pressione ENTER após fazer as alterações ou CTRL+C para cancelar..."
fi

if grep -q "publicacoes->firstItem()" resources/views/publicacoes/index.blade.php; then
    log_success "✅ View de publicações já contém paginação melhorada"
else
    log_warning "⚠️ View de publicações pode precisar de atualização"
fi

# 4. DETECTAR PHP CORRETO
PHP_CMD="php"
if command -v php8.3 &> /dev/null; then
    PHP_CMD="php8.3"
    log_info "Usando PHP 8.3"
elif command -v php8.2 &> /dev/null; then
    PHP_CMD="php8.2"
    log_info "Usando PHP 8.2"
fi

# 5. LIMPAR CACHES
log_info "Limpando caches..."

$PHP_CMD artisan config:clear 2>&1 | grep -v "deprecated\|Composer detected" || true
$PHP_CMD artisan view:clear 2>&1 | grep -v "deprecated\|Composer detected" || true
$PHP_CMD artisan cache:clear 2>&1 | grep -v "deprecated\|Composer detected" || true

log_success "Caches limpos"

# 6. OTIMIZAR PARA PRODUÇÃO
log_info "Otimizando para produção..."

$PHP_CMD artisan config:cache 2>&1 | grep -v "deprecated\|Composer detected" || true
$PHP_CMD artisan route:cache 2>&1 | grep -v "deprecated\|Composer detected" || true
$PHP_CMD artisan view:cache 2>&1 | grep -v "deprecated\|Composer detected" || true

log_success "Otimizações aplicadas"

# 7. VERIFICAR SE É AMBIENTE DOCKER
log_info "Verificando ambiente..."

if [ -f "docker-compose.yml" ]; then
    log_info "Ambiente Docker detectado"

    read -p "Deseja reiniciar os containers? (s/n) " -n 1 -r
    echo
    if [[ $REPLY =~ ^[Ss]$ ]]; then
        log_info "Reiniciando containers..."
        docker-compose restart app
        log_success "Containers reiniciados"
    fi
else
    log_warning "docker-compose.yml não encontrado. Reinicie o PHP-FPM manualmente se necessário."
fi

# 8. TESTE DE CONECTIVIDADE
log_info "Testando conectividade com servidor..."

if curl -s -o /dev/null -w "%{http_code}" "http://$SERVER_IP:$SERVER_PORT" | grep -q "200\|302"; then
    log_success "Servidor respondendo corretamente"
else
    log_warning "Servidor pode não estar acessível em http://$SERVER_IP:$SERVER_PORT"
fi

# 9. RESUMO
echo ""
echo -e "${GREEN}========================================${NC}"
echo -e "${GREEN}  DEPLOY CONCLUÍDO COM SUCESSO!${NC}"
echo -e "${GREEN}========================================${NC}"
echo ""
echo "📋 PRÓXIMOS PASSOS:"
echo ""
echo "1. Acesse: http://$SERVER_IP:$SERVER_PORT/login"
echo "2. Login: linuxewireless@gmail.com"
echo "3. Navegue para: /publicacoes"
echo "4. Verifique:"
echo "   ✅ Texto 'Mostrando X a Y de Z publicações' aparece"
echo "   ✅ Se houver >15 publicações, botões de paginação aparecem"
echo "   ✅ Classes Bootstrap 5 (.page-item, .page-link) presentes"
echo ""
echo "📁 Backup salvo em: $BACKUP_DIR"
echo ""
echo -e "${BLUE}Leia CORRECAO_PAGINACAO.md para mais detalhes${NC}"
echo ""

# Opção de executar teste Dusk
read -p "Deseja executar teste Dusk de validação? (s/n) " -n 1 -r
echo
if [[ $REPLY =~ ^[Ss]$ ]]; then
    log_info "Executando teste Dusk..."
    $PHP_CMD artisan dusk tests/Browser/ProducaoPaginacaoTest.php --filter=testVerificarCorrecaoAplicada 2>&1 | tail -20
fi

log_success "Script finalizado!"
