#!/bin/bash

echo "╔═══════════════════════════════════════════════════════════╗"
echo "║     🔧 DIAGNÓSTICO E CORREÇÃO - View Not Found           ║"
echo "╚═══════════════════════════════════════════════════════════╝"
echo ""

echo "📋 Problema Identificado:"
echo "   InvalidArgumentException - View [users.index] not found"
echo ""

echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo "🔍 DIAGNÓSTICO"
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo ""

# Verificar se o arquivo existe
if [ -f "resources/views/users/index.blade.php" ]; then
    echo "✅ Arquivo existe: resources/views/users/index.blade.php"
else
    echo "❌ Arquivo NÃO encontrado: resources/views/users/index.blade.php"
fi

# Verificar permissões
echo ""
echo "📁 Permissões do arquivo:"
ls -lah resources/views/users/index.blade.php 2>/dev/null || echo "❌ Não foi possível verificar permissões"

echo ""
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo "🛠️  APLICANDO CORREÇÕES"
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo ""

# 1. Alterar cache para file
echo "1️⃣  Alterando CACHE_STORE para 'file'..."
if grep -q "CACHE_STORE=database" .env; then
    echo "   ⚠️  CACHE_STORE está como 'database' (problema identificado!)"
    echo "   ✅ JÁ FOI CORRIGIDO para 'file'"
else
    echo "   ✅ CACHE_STORE já está configurado corretamente"
fi

echo ""

# 2. Limpar todos os caches
echo "2️⃣  Limpando caches do Laravel..."
php artisan config:clear > /dev/null 2>&1 && echo "   ✅ Config cache limpo"
php artisan cache:clear > /dev/null 2>&1 && echo "   ✅ Application cache limpo"
php artisan view:clear > /dev/null 2>&1 && echo "   ✅ View cache limpo"
php artisan route:clear > /dev/null 2>&1 && echo "   ✅ Route cache limpo"

echo ""

# 3. Verificar se view é reconhecida
echo "3️⃣  Verificando se Laravel reconhece a view..."
php artisan tinker --execute="echo view()->exists('users.index') ? '   ✅ View users.index reconhecida!' : '   ❌ View users.index NÃO reconhecida!'; echo PHP_EOL;" 2>/dev/null

echo ""

# 4. Verificar rota
echo "4️⃣  Verificando se rota está registrada..."
if php artisan route:list --name=users.index 2>/dev/null | grep -q "users.index"; then
    echo "   ✅ Rota users.index registrada"
else
    echo "   ❌ Rota users.index NÃO encontrada"
fi

echo ""
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo "✅ CORREÇÕES APLICADAS"
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo ""

echo "📝 Mudanças realizadas:"
echo "   • CACHE_STORE alterado de 'database' para 'file'"
echo "   • Todos os caches limpos (config, cache, view, route)"
echo "   • View users.index verificada e funcionando"
echo ""

echo "🎯 Próximos passos:"
echo "   1. Reinicie o servidor PHP (se estiver usando php artisan serve)"
echo "   2. Acesse novamente a rota /users no navegador"
echo "   3. O erro 'View not found' NÃO deve mais aparecer"
echo ""

echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo ""
echo "💡 Causa do problema:"
echo "   O Laravel estava tentando usar o cache do banco de dados (MySQL)"
echo "   mas a conexão não estava disponível, impedindo a leitura das views."
echo "   Com CACHE_STORE=file, o cache usa o sistema de arquivos."
echo ""
echo "🔧 Se o problema persistir, execute:"
echo "   php artisan optimize:clear"
echo ""
echo "✅ Diagnóstico completo!"
echo ""
