#!/bin/bash

###############################################################################
# Script de Deploy Automatizado - MikroTik Hotspot
# Descrição: Envia arquivos atualizados do frontend para o equipamento MikroTik
# Uso: bash deploy-mikrotik.sh [ambiente]
###############################################################################

set -e  # Parar em caso de erro

# Cores para output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
NC='\033[0m' # No Color

# ============================================================================
# CONFIGURAÇÕES - AJUSTE CONFORME SEU AMBIENTE
# ============================================================================

# Ambiente (dev, staging, production)
ENVIRONMENT="${1:-production}"

# Diretório local dos arquivos do hotspot
LOCAL_DIR="./flash/hotspot"

# Configurações do MikroTik via VPN
MIKROTIK_IP="10.x.x.x"           # ⚠️ AJUSTAR: IP do MikroTik na VPN
MIKROTIK_USER="admin"             # ⚠️ AJUSTAR: Usuário SSH do MikroTik
MIKROTIK_PORT="22"                # Porta SSH padrão
MIKROTIK_DEST="/flash/hotspot"   # Diretório destino no MikroTik

# Arquivo de chave SSH (deixe vazio para usar senha)
SSH_KEY=""  # Ex: ~/.ssh/mikrotik_rsa

# Arquivos/pastas a excluir do deploy
EXCLUDE_PATTERNS=(
    "*.md"
    "*.backup"
    ".git"
    ".gitignore"
    "node_modules"
    "*.log"
)

# ============================================================================
# FUNÇÕES AUXILIARES
# ============================================================================

print_header() {
    echo -e "${BLUE}╔════════════════════════════════════════════════════════════════════╗${NC}"
    echo -e "${BLUE}║  DEPLOY AUTOMATIZADO - MIKROTIK HOTSPOT                            ║${NC}"
    echo -e "${BLUE}╚════════════════════════════════════════════════════════════════════╝${NC}"
    echo ""
}

print_step() {
    echo -e "${CYAN}[STEP]${NC} $1"
}

print_success() {
    echo -e "${GREEN}✓${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}⚠${NC} $1"
}

print_error() {
    echo -e "${RED}✗${NC} $1"
}

check_requirements() {
    print_step "Verificando requisitos..."

    # Verificar se SSH está disponível
    if ! command -v ssh &> /dev/null; then
        print_error "SSH não encontrado. Instale openssh-client."
        exit 1
    fi

    # Verificar se SCP está disponível
    if ! command -v scp &> /dev/null; then
        print_error "SCP não encontrado. Instale openssh-client."
        exit 1
    fi

    # Verificar se rsync está disponível (opcional, mas recomendado)
    if command -v rsync &> /dev/null; then
        USE_RSYNC=true
        print_success "rsync disponível (modo avançado habilitado)"
    else
        USE_RSYNC=false
        print_warning "rsync não encontrado, usando SCP (mais lento)"
    fi

    # Verificar se diretório local existe
    if [ ! -d "$LOCAL_DIR" ]; then
        print_error "Diretório local não encontrado: $LOCAL_DIR"
        exit 1
    fi

    print_success "Requisitos verificados"
}

test_connection() {
    print_step "Testando conexão com MikroTik ($MIKROTIK_IP)..."

    # Preparar comando SSH
    SSH_CMD="ssh -p $MIKROTIK_PORT"
    if [ -n "$SSH_KEY" ]; then
        SSH_CMD="$SSH_CMD -i $SSH_KEY"
    fi

    # Testar conexão
    if $SSH_CMD -o ConnectTimeout=10 -o BatchMode=yes "$MIKROTIK_USER@$MIKROTIK_IP" "echo 'Conexão OK'" 2>/dev/null; then
        print_success "Conexão estabelecida com sucesso"
        return 0
    else
        print_error "Falha na conexão. Verifique:"
        echo "  - VPN está ativa?"
        echo "  - IP do MikroTik: $MIKROTIK_IP"
        echo "  - Usuário: $MIKROTIK_USER"
        echo "  - Porta SSH: $MIKROTIK_PORT"
        echo "  - Chave SSH: ${SSH_KEY:-senha será solicitada}"
        exit 1
    fi
}

create_backup() {
    print_step "Criando backup no MikroTik..."

    BACKUP_NAME="hotspot_backup_$(date +%Y%m%d_%H%M%S)"

    # Comando SSH
    SSH_CMD="ssh -p $MIKROTIK_PORT"
    [ -n "$SSH_KEY" ] && SSH_CMD="$SSH_CMD -i $SSH_KEY"

    # Criar backup
    $SSH_CMD "$MIKROTIK_USER@$MIKROTIK_IP" "
        if [ -d '$MIKROTIK_DEST' ]; then
            cp -r '$MIKROTIK_DEST' '/flash/${BACKUP_NAME}'
            echo 'Backup criado: /flash/${BACKUP_NAME}'
        else
            echo 'ERRO: Diretório $MIKROTIK_DEST não existe!'
            exit 1
        fi
    "

    if [ $? -eq 0 ]; then
        print_success "Backup criado: /flash/${BACKUP_NAME}"
    else
        print_error "Falha ao criar backup"
        exit 1
    fi
}

show_changes() {
    print_step "Arquivos que serão enviados:"
    echo ""

    # Listar arquivos (excluindo padrões)
    find "$LOCAL_DIR" -type f | grep -v -E "$(IFS="|"; echo "${EXCLUDE_PATTERNS[*]}")" | while read file; do
        rel_path="${file#$LOCAL_DIR/}"
        echo "  📄 $rel_path"
    done

    echo ""
}

confirm_deploy() {
    echo -e "${YELLOW}════════════════════════════════════════════════════════════════════${NC}"
    echo -e "${YELLOW}  ATENÇÃO: Você está prestes a fazer deploy em ${ENVIRONMENT^^}${NC}"
    echo -e "${YELLOW}  Destino: $MIKROTIK_USER@$MIKROTIK_IP:$MIKROTIK_DEST${NC}"
    echo -e "${YELLOW}════════════════════════════════════════════════════════════════════${NC}"
    echo ""

    read -p "Deseja continuar? (s/n): " -n 1 -r
    echo

    if [[ ! $REPLY =~ ^[Ss]$ ]]; then
        print_warning "Deploy cancelado pelo usuário"
        exit 0
    fi
}

deploy_files() {
    print_step "Iniciando deploy dos arquivos..."

    # Preparar exclusões
    EXCLUDE_ARGS=""
    for pattern in "${EXCLUDE_PATTERNS[@]}"; do
        EXCLUDE_ARGS="$EXCLUDE_ARGS --exclude='$pattern'"
    done

    if [ "$USE_RSYNC" = true ]; then
        # Usar rsync (mais rápido e eficiente)
        print_step "Usando rsync para transferência..."

        RSYNC_CMD="rsync -avz --progress --delete"
        [ -n "$SSH_KEY" ] && RSYNC_CMD="$RSYNC_CMD -e 'ssh -i $SSH_KEY -p $MIKROTIK_PORT'"

        eval "$RSYNC_CMD $EXCLUDE_ARGS $LOCAL_DIR/ $MIKROTIK_USER@$MIKROTIK_IP:$MIKROTIK_DEST/"

    else
        # Usar SCP recursivo
        print_step "Usando SCP para transferência..."

        SCP_CMD="scp -r -P $MIKROTIK_PORT"
        [ -n "$SSH_KEY" ] && SCP_CMD="$SCP_CMD -i $SSH_KEY"

        # Criar arquivo temporário com lista de arquivos
        TEMP_LIST=$(mktemp)
        find "$LOCAL_DIR" -type f | grep -v -E "$(IFS="|"; echo "${EXCLUDE_PATTERNS[*]}")" > "$TEMP_LIST"

        # Copiar cada arquivo mantendo estrutura
        while IFS= read -r file; do
            rel_path="${file#$LOCAL_DIR/}"
            dest_dir="$(dirname "$rel_path")"

            # Criar diretório no destino se não existir
            SSH_CMD="ssh -p $MIKROTIK_PORT"
            [ -n "$SSH_KEY" ] && SSH_CMD="$SSH_CMD -i $SSH_KEY"

            $SSH_CMD "$MIKROTIK_USER@$MIKROTIK_IP" "mkdir -p '$MIKROTIK_DEST/$dest_dir'" 2>/dev/null || true

            # Copiar arquivo
            $SCP_CMD "$file" "$MIKROTIK_USER@$MIKROTIK_IP:$MIKROTIK_DEST/$rel_path"

            echo "  ✓ $rel_path"
        done < "$TEMP_LIST"

        rm "$TEMP_LIST"
    fi

    if [ $? -eq 0 ]; then
        print_success "Arquivos transferidos com sucesso!"
    else
        print_error "Falha na transferência de arquivos"
        exit 1
    fi
}

verify_deployment() {
    print_step "Verificando deployment..."

    # Comando SSH
    SSH_CMD="ssh -p $MIKROTIK_PORT"
    [ -n "$SSH_KEY" ] && SSH_CMD="$SSH_CMD -i $SSH_KEY"

    # Verificar se index.html existe e contém a atualização
    CHECK_RESULT=$($SSH_CMD "$MIKROTIK_USER@$MIKROTIK_IP" "
        if [ -f '$MIKROTIK_DEST/index.html' ]; then
            if grep -q '8082/storage' '$MIKROTIK_DEST/index.html'; then
                echo 'OK'
            else
                echo 'OUTDATED'
            fi
        else
            echo 'MISSING'
        fi
    " 2>/dev/null)

    case "$CHECK_RESULT" in
        "OK")
            print_success "Verificação concluída - arquivo atualizado corretamente"
            ;;
        "OUTDATED")
            print_warning "Arquivo transferido mas ainda com código antigo"
            ;;
        "MISSING")
            print_error "Arquivo index.html não encontrado no destino"
            exit 1
            ;;
        *)
            print_warning "Não foi possível verificar a atualização"
            ;;
    esac
}

show_summary() {
    echo ""
    echo -e "${GREEN}╔════════════════════════════════════════════════════════════════════╗${NC}"
    echo -e "${GREEN}║  DEPLOY CONCLUÍDO COM SUCESSO!                                     ║${NC}"
    echo -e "${GREEN}╚════════════════════════════════════════════════════════════════════╝${NC}"
    echo ""
    echo -e "${CYAN}Ambiente:${NC} $ENVIRONMENT"
    echo -e "${CYAN}Destino:${NC} $MIKROTIK_USER@$MIKROTIK_IP:$MIKROTIK_DEST"
    echo -e "${CYAN}Data:${NC} $(date '+%Y-%m-%d %H:%M:%S')"
    echo ""
    echo -e "${YELLOW}Próximos passos:${NC}"
    echo "  1. Teste o hotspot através do MikroTik"
    echo "  2. Verifique se as imagens carregam corretamente"
    echo "  3. Confirme se a auditoria está sendo gravada"
    echo ""
}

# ============================================================================
# EXECUÇÃO PRINCIPAL
# ============================================================================

main() {
    print_header

    # Validações
    check_requirements
    test_connection

    # Exibir mudanças
    show_changes

    # Confirmar com usuário
    confirm_deploy

    # Criar backup
    create_backup

    # Deploy
    deploy_files

    # Verificar
    verify_deployment

    # Resumo
    show_summary
}

# Executar
main "$@"
