/**
 * Modal Preview - Solução JavaScript Inline
 * Aplica estilos diretamente via JavaScript para garantir que a imagem
 * NUNCA ultrapasse o tamanho do modal, independente do CSS do Bootstrap
 */

document.addEventListener('DOMContentLoaded', function () {
    const previewModal = document.getElementById('previewModal');
    
    if (!previewModal) {
        console.warn('⚠️ Modal #previewModal não encontrado');
        return;
    }
    
    console.log('✅ Modal Preview JS carregado');
    
    /**
     * Função para calcular e aplicar estilos FORÇADOS no modal
     * MODAL COMO MOLDURA - ajusta ao tamanho da imagem, sempre centralizado
     */
    function forceModalStyles() {
        const modalDialog = previewModal.querySelector('.modal-dialog');
        const modalContent = previewModal.querySelector('.modal-content');
        const modalBody = previewModal.querySelector('.modal-body');
        const modalHeader = previewModal.querySelector('.modal-header');
        
        console.log('📏 Aplicando estilos (moldura centralizada)');
        
        if (modalDialog) {
            // Modal se ajusta ao conteúdo (moldura) mas centralizado
            modalDialog.style.setProperty('max-width', '50vw', 'important'); // Máximo 90% da tela
            modalDialog.style.setProperty('width', 'fit-content', 'important'); // Ajusta ao conteúdo!
            modalDialog.style.setProperty('margin', '1.75rem auto', 'important'); // Centralizado
            modalDialog.style.setProperty('display', 'block', 'important'); // Block para centralizar
        }
        
        if (modalContent) {
            // Modal-content como moldura justa
            modalContent.style.setProperty('width', 'fit-content', 'important'); // Moldura justa
            modalContent.style.setProperty('margin', '0', 'important');
            modalContent.style.setProperty('box-sizing', 'border-box', 'important');
        }
        
        if (modalHeader) {
            // Header compacto
            modalHeader.style.setProperty('padding', '0.75rem 1rem', 'important');
        }
        
        if (modalBody) {
            // Modal-body com padding mínimo (moldura bonita mas justa)
            modalBody.style.setProperty('width', 'auto', 'important');
            modalBody.style.setProperty('padding', '1rem', 'important'); // Padding mínimo = moldura
            modalBody.style.setProperty('display', 'block', 'important');
            modalBody.style.setProperty('min-height', 'auto', 'important'); // Sem altura mínima
        }
    }
    
    /**
     * Função para aplicar estilos FORÇADOS na imagem
     */
    function forceImageStyles(img) {
        if (!img) return;
        
        // Aplicar estilos inline na imagem usando setProperty com !important
        img.style.setProperty('max-width', '50%', 'important');
        img.style.setProperty('max-height', '70vh', 'important');
        img.style.setProperty('width', 'auto', 'important');
        img.style.setProperty('height', 'auto', 'important');
        img.style.setProperty('display', 'block', 'important');
        img.style.setProperty('margin', '0 auto', 'important');
        img.style.setProperty('object-fit', 'contain', 'important');
        img.style.setProperty('border-radius', '8px', 'important');
        img.style.setProperty('box-shadow', '0 10px 40px rgba(0, 0, 0, 0.5)', 'important');
        
        console.log('🖼️ Estilos aplicados na imagem:', {
            naturalWidth: img.naturalWidth,
            naturalHeight: img.naturalHeight,
            displayWidth: img.offsetWidth,
            displayHeight: img.offsetHeight
        });
    }
    
    /**
     * Evento: Quando o modal começa a abrir
     */
    previewModal.addEventListener('show.bs.modal', function (event) {
        const button = event.relatedTarget;
        const mediaPath = button.getAttribute('data-path');
        const mediaTitle = button.getAttribute('data-title');
        const mediaType = button.getAttribute('data-type');
        const modalBody = document.getElementById('previewBody');
        const modalTitle = document.getElementById('previewTitle');
        
        console.log('🚀 Modal abrindo:', { mediaPath, mediaTitle, mediaType });
        
        // Atualizar título com cor VISÍVEL
        modalTitle.textContent = mediaTitle || 'Preview do Criativo';
        modalTitle.style.setProperty('color', '#000000', 'important'); // Texto preto
        modalTitle.style.setProperty('font-weight', '600', 'important');
        modalTitle.style.setProperty('font-size', '1.25rem', 'important');
        
        // Mostrar preloader
        modalBody.innerHTML = `
            <div class="preview-loader">
                <i class="ph-duotone ph-circle-notch ph-spin f-s-48 text-white mb-3"></i>
                <p class="text-white">Carregando preview...</p>
            </div>
        `;
        
        // Criar e carregar a imagem
        setTimeout(function() {
            if (mediaType === 'imagem' || mediaType === 'image') {
                const img = new Image();
                img.src = mediaPath;
                img.alt = mediaTitle;
                
                // Quando a imagem carregar
                img.onload = function() {
                    console.log('✅ Imagem carregada');
                    
                    // Limpar modal body
                    modalBody.innerHTML = '';
                    
                    // Aplicar estilos ANTES de inserir
                    forceImageStyles(img);
                    
                    // Inserir imagem
                    modalBody.appendChild(img);
                    
                    // AGORA SIM: Expandir modal para se ajustar à imagem (CENTRALIZADO)
                    setTimeout(() => {
                        forceImageStyles(img);
                        forceModalStyles(); // Expande e mantém centralizado
                    }, 50);
                    
                    // RE-APLICAR novamente após 200ms (garantia tripla)
                    setTimeout(() => {
                        forceImageStyles(img);
                        forceModalStyles();
                    }, 200);
                    
                    // Adicionar animação
                    img.style.setProperty('animation', 'fadeIn 0.3s ease-in', 'important');
                };
                
                img.onerror = function() {
                    console.error('❌ Erro ao carregar imagem');
                    modalBody.innerHTML = '<div class="alert alert-danger text-white">Erro ao carregar imagem</div>';
                };
                
            } else if (mediaType === 'video') {
                const video = document.createElement('video');
                video.src = mediaPath;
                video.controls = true;
                
                // Aplicar estilos no vídeo
                video.style.setProperty('max-width', '100%', 'important');
                video.style.setProperty('max-height', '70vh', 'important');
                video.style.setProperty('width', 'auto', 'important');
                video.style.setProperty('height', 'auto', 'important');
                video.style.setProperty('border-radius', '8px', 'important');
                video.style.setProperty('object-fit', 'contain', 'important');
                
                modalBody.innerHTML = '';
                modalBody.appendChild(video);
                
                setTimeout(() => forceModalStyles(), 50);
            }
        }, 100);
    });
    
    /**
     * Evento: Quando o modal terminou de abrir (totalmente visível)
     */
    previewModal.addEventListener('shown.bs.modal', function() {
        console.log('✅ Modal totalmente aberto');
        
        // Forçar estilos novamente
        forceModalStyles();
        
        // Re-aplicar estilos na imagem
        const img = previewModal.querySelector('.modal-body img');
        if (img) {
            forceImageStyles(img);
        }
    });
    
    /**
     * Evento: Redimensionamento da janela
     */
    let resizeTimer;
    window.addEventListener('resize', function() {
        clearTimeout(resizeTimer);
        resizeTimer = setTimeout(function() {
            if (previewModal.classList.contains('show')) {
                console.log('🔄 Janela redimensionada, reaplicando estilos');
                forceModalStyles();
                
                const img = previewModal.querySelector('.modal-body img');
                if (img) {
                    forceImageStyles(img);
                }
            }
        }, 250);
    });
    
    console.log('✅ Listeners do modal configurados');
});
