#!/bin/bash

###############################################################################
# Script de Setup do Servidor de Produção - FreeFi Admin
# Servidor: 172.233.21.57 (Linode/Akamai)
###############################################################################

set -e

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

print_error() { echo -e "${RED}✗ $1${NC}"; }
print_success() { echo -e "${GREEN}✓ $1${NC}"; }
print_warning() { echo -e "${YELLOW}⚠ $1${NC}"; }
print_info() { echo -e "${BLUE}ℹ $1${NC}"; }

echo "╔════════════════════════════════════════════════════════════════════╗"
echo "║  SETUP DO SERVIDOR DE PRODUÇÃO - FREEFI ADMIN                     ║"
echo "║  Servidor: 172.233.21.57                                          ║"
echo "╚════════════════════════════════════════════════════════════════════╝"
echo ""

# Verificar se está rodando como root
if [ "$EUID" -ne 0 ]; then
    print_error "Este script deve ser executado como root"
    echo "Execute: sudo bash $0"
    exit 1
fi

print_info "Este script irá:"
echo "  1. Criar diretório do projeto em /var/www/freefi"
echo "  2. Clonar o repositório do GitHub"
echo "  3. Configurar arquivo .env para produção"
echo "  4. Ajustar permissões"
echo "  5. Verificar rede Docker root_mikrotik_net"
echo ""

read -p "Deseja continuar? (s/n): " continuar
if [ "$continuar" != "s" ] && [ "$continuar" != "S" ]; then
    print_warning "Operação cancelada."
    exit 0
fi

echo ""
print_info "PASSO 1/6: Verificando Docker..."

if ! command -v docker &> /dev/null; then
    print_error "Docker não está instalado!"
    exit 1
fi

DOCKER_VERSION=$(docker --version)
print_success "Docker instalado: $DOCKER_VERSION"

if ! command -v docker compose &> /dev/null; then
    print_error "Docker Compose não está instalado!"
    exit 1
fi

print_success "Docker Compose instalado"

echo ""
print_info "PASSO 2/6: Verificando rede Docker root_mikrotik_net..."

if docker network inspect root_mikrotik_net &> /dev/null; then
    print_success "Rede root_mikrotik_net encontrada"
else
    print_error "Rede root_mikrotik_net não encontrada!"
    print_info "Criando rede root_mikrotik_net..."
    docker network create root_mikrotik_net
    print_success "Rede criada com sucesso"
fi

echo ""
print_info "PASSO 3/6: Criando diretório do projeto..."

PROJECT_DIR="/var/www/freefi"

if [ -d "$PROJECT_DIR" ]; then
    print_warning "Diretório $PROJECT_DIR já existe!"
    read -p "Deseja fazer backup e recriar? (s/n): " backup
    if [ "$backup" = "s" ] || [ "$backup" = "S" ]; then
        BACKUP_DIR="${PROJECT_DIR}_backup_$(date +%Y%m%d_%H%M%S)"
        print_info "Fazendo backup para $BACKUP_DIR..."
        mv "$PROJECT_DIR" "$BACKUP_DIR"
        print_success "Backup criado"
    else
        print_info "Usando diretório existente"
    fi
else
    mkdir -p "$PROJECT_DIR"
    print_success "Diretório criado: $PROJECT_DIR"
fi

cd "$PROJECT_DIR"

echo ""
print_info "PASSO 4/6: Clonando repositório..."

if [ -d ".git" ]; then
    print_info "Repositório já existe. Fazendo pull..."
    git pull origin main
else
    print_info "Clonando repositório..."

    # Tentar com SSH primeiro
    if git clone git@github.com:freefi-com-br/freefiadm.git . 2>/dev/null; then
        print_success "Repositório clonado via SSH"
    else
        print_warning "SSH falhou, tentando HTTPS..."
        # Tentar com HTTPS
        if git clone https://github.com/freefi-com-br/freefiadm.git .; then
            print_success "Repositório clonado via HTTPS"
        else
            print_error "Falha ao clonar repositório!"
            print_info "Configure suas credenciais do GitHub e tente novamente"
            exit 1
        fi
    fi
fi

echo ""
print_info "PASSO 5/6: Configurando arquivo .env..."

if [ -f ".env" ]; then
    print_warning "Arquivo .env já existe"
    read -p "Deseja sobrescrever? (s/n): " sobrescrever
    if [ "$sobrescrever" != "s" ] && [ "$sobrescrever" != "S" ]; then
        print_info "Mantendo .env existente"
    else
        cp .env.production.example .env
        print_success "Arquivo .env criado a partir de .env.production.example"
        print_warning "IMPORTANTE: Edite o arquivo .env e configure:"
        echo "  - APP_KEY (gerar com: docker compose run --rm app php artisan key:generate)"
        echo "  - DB_PASSWORD (senha segura para o banco)"
        echo "  - DB_ROOT_PASSWORD (senha root do MySQL)"
        echo "  - MAIL_USERNAME e MAIL_PASSWORD (credenciais SMTP)"
    fi
else
    if [ -f ".env.production.example" ]; then
        cp .env.production.example .env
        print_success "Arquivo .env criado a partir de .env.production.example"
    else
        cp .env.example .env
        print_success "Arquivo .env criado a partir de .env.example"
    fi

    print_warning "IMPORTANTE: Configure o arquivo .env antes de continuar!"
    echo ""
    read -p "Deseja editar o .env agora? (s/n): " editar
    if [ "$editar" = "s" ] || [ "$editar" = "S" ]; then
        nano .env
    fi
fi

echo ""
print_info "PASSO 6/6: Configurando permissões..."

# Criar diretórios necessários
mkdir -p storage/framework/{sessions,views,cache}
mkdir -p storage/logs
mkdir -p bootstrap/cache

# Ajustar permissões
chown -R www-data:www-data storage bootstrap/cache
chmod -R 775 storage bootstrap/cache

print_success "Permissões configuradas"

echo ""
echo "════════════════════════════════════════════════════════════════════"
print_success "SETUP CONCLUÍDO!"
echo "════════════════════════════════════════════════════════════════════"
echo ""

print_info "Próximos passos:"
echo ""
echo "1. Gerar APP_KEY:"
echo "   cd $PROJECT_DIR"
echo "   docker compose run --rm app php artisan key:generate"
echo ""
echo "2. Subir os containers:"
echo "   docker compose up -d"
echo ""
echo "3. Rodar migrations:"
echo "   docker compose exec app php artisan migrate --force"
echo ""
echo "4. Criar link do storage:"
echo "   docker compose exec app php artisan storage:link"
echo ""
echo "5. Testar aplicação:"
echo "   curl http://localhost:8082"
echo "   ou"
echo "   curl http://172.233.21.57:8082"
echo ""
echo "6. Ver logs:"
echo "   docker compose logs -f"
echo ""

print_warning "LEMBRE-SE:"
echo "  - Configure o arquivo .env com credenciais SEGURAS"
echo "  - APP_DEBUG deve ser false em produção"
echo "  - Não use as credenciais expostas no código!"
echo ""
