#!/bin/bash
# Script de teste de integração Mikrotik → Laravel
# Execute este script para simular chamadas que o Mikrotik fará

API_URL="http://localhost/api/hotspot"
HOTSPOT_SERIAL="HS-SHOPPING-001"
HOTSPOT_ID=3

echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo "🧪 SIMULADOR DE INTEGRAÇÃO MIKROTIK → LARAVEL"
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo ""
echo "📡 API Base: $API_URL"
echo "🔧 Hotspot: $HOTSPOT_SERIAL (ID: $HOTSPOT_ID)"
echo ""

# Cores para output
GREEN='\033[0;32m'
RED='\033[0;31m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Função de teste
test_endpoint() {
    local name=$1
    local method=$2
    local endpoint=$3
    local data=$4
    
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
    echo -e "${YELLOW}🔍 Testando: $name${NC}"
    echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
    echo "Método: $method"
    echo "Endpoint: $endpoint"
    if [ -n "$data" ]; then
        echo "Dados: $data"
    fi
    echo ""
    
    if [ "$method" == "GET" ]; then
        response=$(curl -s -w "\n%{http_code}" "$endpoint")
    else
        response=$(curl -s -w "\n%{http_code}" -X "$method" "$endpoint" \
            -H "Content-Type: application/json" \
            -d "$data")
    fi
    
    http_code=$(echo "$response" | tail -n1)
    body=$(echo "$response" | sed '$d')
    
    if [ "$http_code" == "200" ] || [ "$http_code" == "201" ]; then
        echo -e "${GREEN}✅ Sucesso (HTTP $http_code)${NC}"
        echo "$body" | jq '.' 2>/dev/null || echo "$body"
    else
        echo -e "${RED}❌ Erro (HTTP $http_code)${NC}"
        echo "$body"
    fi
    echo ""
}

# ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
# 1. HEARTBEAT
# ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
test_endpoint \
    "1. Heartbeat (Mikrotik avisa que está online)" \
    "POST" \
    "$API_URL/heartbeat" \
    "{\"serial\":\"$HOTSPOT_SERIAL\",\"vpn_ip\":\"10.0.0.100\"}"

sleep 1

# ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
# 2. BUSCAR PUBLICAÇÕES
# ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
test_endpoint \
    "2. Buscar Publicações Vigentes" \
    "GET" \
    "$API_URL/$HOTSPOT_SERIAL/publicacoes"

sleep 1

# ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
# 3. BUSCAR WALLED GARDENS
# ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
test_endpoint \
    "3. Buscar Walled Gardens da Publicação 1" \
    "GET" \
    "$API_URL/publicacao/1/walled-gardens"

sleep 1

# ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
# 4. REGISTRAR IMPRESSÃO
# ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
test_endpoint \
    "4. Registrar Impressão (Usuário viu o banner)" \
    "POST" \
    "$API_URL/impressao" \
    "{\"publicacao_id\":1,\"hotspot_id\":$HOTSPOT_ID,\"criativo_id\":1,\"data\":\"$(date +%Y-%m-%d)\",\"impressoes_unicas\":5}"

sleep 1

# ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
# 5. REGISTRAR CLIQUE
# ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
test_endpoint \
    "5. Registrar Clique (Usuário clicou no banner)" \
    "POST" \
    "$API_URL/clique" \
    "{\"publicacao_id\":1,\"hotspot_id\":$HOTSPOT_ID,\"criativo_id\":1,\"user_ip\":\"192.168.1.100\",\"user_mac\":\"AA:BB:CC:DD:EE:FF\"}"

sleep 1

# ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
# RESUMO
# ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo -e "${GREEN}✅ Simulação Concluída!${NC}"
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo ""
echo "📊 Próximos Passos:"
echo "1. Verificar se dados foram salvos no banco:"
echo "   docker-compose exec app php artisan tinker"
echo "   >>> App\\Models\\PublicacaoImpressao::count();"
echo "   >>> App\\Models\\PublicacaoClique::count();"
echo ""
echo "2. Acessar dashboard para ver estatísticas:"
echo "   http://localhost/publicacoes/1/estatisticas"
echo ""
echo "3. Implementar scripts no Mikrotik (ver INTEGRACAO_MIKROTIK.md)"
echo ""
