<?php

namespace Tests\Browser;

use App\Models\User;
use App\Models\Empresa;
use App\Models\Publicacao;
use App\Models\Criativo;
use Illuminate\Foundation\Testing\DatabaseMigrations;
use Laravel\Dusk\Browser;
use Tests\DuskTestCase;

class PublicacaoAnalyticsModalTest extends DuskTestCase
{
    use DatabaseMigrations;

    /**
     * Teste: Modal de Analytics abre ao clicar no ícone
     */
    public function testModalAnalyticsAbreAoClicarNoIcone()
    {
        $this->browse(function (Browser $browser) {
            // Criar dados de teste
            $user = $this->criarUsuarioAutenticado();
            $publicacao = $this->criarPublicacao();

            // Fazer login e visitar a página de publicações
            $browser->loginAs($user)
                    ->visit('/publicacoes')
                    ->assertSee('Lista de Publicações')
                    ->waitFor('table', 5);

            // Verificar que o modal não está visível inicialmente
            $browser->assertDontSee('Visualizações da publicação');

            // Clicar no ícone de analytics (primeiro ícone de gráfico na tabela)
            $browser->click('[data-bs-toggle="modal"][data-bs-target="#analyticsModal"]')
                    ->waitFor('#analyticsModal.show', 5);

            // Verificar que o modal está visível
            $browser->assertVisible('#analyticsModal')
                    ->assertSee('Visualizações da publicação');
        });
    }

    /**
     * Teste: Título do modal é atualizado com o nome da publicação
     */
    public function testTituloModalEAtualizadoComNomeDaPublicacao()
    {
        $this->browse(function (Browser $browser) {
            // Criar dados de teste
            $user = $this->criarUsuarioAutenticado();
            $publicacao = $this->criarPublicacao([
                'titulo' => 'Black Friday 2025 - 50% OFF'
            ]);

            // Fazer login e visitar a página
            $browser->loginAs($user)
                    ->visit('/publicacoes')
                    ->waitFor('table', 5);

            // Clicar no ícone específico desta publicação
            $browser->click('[data-publicacao-id="' . $publicacao->id . '"]')
                    ->waitFor('#analyticsModal.show', 5);

            // Verificar que o título contém o nome da publicação
            $browser->assertSeeIn('#analyticsModalLabel', 'Visualizações: Black Friday 2025 - 50% OFF');
        });
    }

    /**
     * Teste: Modal pode ser fechado clicando no botão X
     */
    public function testModalPodeSerFechadoPeloBotaoX()
    {
        $this->browse(function (Browser $browser) {
            // Criar dados de teste
            $user = $this->criarUsuarioAutenticado();
            $publicacao = $this->criarPublicacao();

            // Fazer login e abrir o modal
            $browser->loginAs($user)
                    ->visit('/publicacoes')
                    ->waitFor('table', 5)
                    ->click('[data-bs-toggle="modal"][data-bs-target="#analyticsModal"]')
                    ->waitFor('#analyticsModal.show', 5);

            // Verificar que o modal está aberto
            $browser->assertVisible('#analyticsModal');

            // Clicar no botão de fechar (X)
            $browser->click('#analyticsModal .btn-close')
                    ->waitUntilMissing('#analyticsModal.show', 5);

            // Verificar que o modal foi fechado
            $browser->assertNotPresent('#analyticsModal.show');
        });
    }

    /**
     * Teste: Modal pode ser fechado clicando fora dele (backdrop)
     */
    public function testModalPodeSerFechadoClicandoForaDoModal()
    {
        $this->browse(function (Browser $browser) {
            // Criar dados de teste
            $user = $this->criarUsuarioAutenticado();
            $publicacao = $this->criarPublicacao();

            // Fazer login e abrir o modal
            $browser->loginAs($user)
                    ->visit('/publicacoes')
                    ->waitFor('table', 5)
                    ->click('[data-bs-toggle="modal"][data-bs-target="#analyticsModal"]')
                    ->waitFor('#analyticsModal.show', 5);

            // Verificar que o modal está aberto
            $browser->assertVisible('#analyticsModal');

            // Clicar no backdrop (fora do modal)
            $browser->script("document.querySelector('#analyticsModal').click()");

            // Aguardar o modal fechar
            $browser->pause(500)
                    ->waitUntilMissing('#analyticsModal.show', 5);
        });
    }

    /**
     * Teste: Múltiplas publicações exibem o modal corretamente
     */
    public function testMultiplasPublicacoesExibemModalCorretamente()
    {
        $this->browse(function (Browser $browser) {
            // Criar dados de teste
            $user = $this->criarUsuarioAutenticado();
            $publicacao1 = $this->criarPublicacao(['titulo' => 'Campanha 1']);
            $publicacao2 = $this->criarPublicacao(['titulo' => 'Campanha 2']);
            $publicacao3 = $this->criarPublicacao(['titulo' => 'Campanha 3']);

            // Fazer login
            $browser->loginAs($user)
                    ->visit('/publicacoes')
                    ->waitFor('table', 5);

            // Testar primeira publicação
            $browser->click('[data-publicacao-id="' . $publicacao1->id . '"]')
                    ->waitFor('#analyticsModal.show', 5)
                    ->assertSeeIn('#analyticsModalLabel', 'Visualizações: Campanha 1')
                    ->click('#analyticsModal .btn-close')
                    ->waitUntilMissing('#analyticsModal.show', 5);

            // Testar segunda publicação
            $browser->click('[data-publicacao-id="' . $publicacao2->id . '"]')
                    ->waitFor('#analyticsModal.show', 5)
                    ->assertSeeIn('#analyticsModalLabel', 'Visualizações: Campanha 2')
                    ->click('#analyticsModal .btn-close')
                    ->waitUntilMissing('#analyticsModal.show', 5);

            // Testar terceira publicação
            $browser->click('[data-publicacao-id="' . $publicacao3->id . '"]')
                    ->waitFor('#analyticsModal.show', 5)
                    ->assertSeeIn('#analyticsModalLabel', 'Visualizações: Campanha 3');
        });
    }

    /**
     * Teste: Conteúdo do modal exibe mensagem placeholder
     */
    public function testConteudoModalExibeMensagemPlaceholder()
    {
        $this->browse(function (Browser $browser) {
            // Criar dados de teste
            $user = $this->criarUsuarioAutenticado();
            $publicacao = $this->criarPublicacao();

            // Fazer login e abrir modal
            $browser->loginAs($user)
                    ->visit('/publicacoes')
                    ->waitFor('table', 5)
                    ->click('[data-bs-toggle="modal"][data-bs-target="#analyticsModal"]')
                    ->waitFor('#analyticsModal.show', 5);

            // Verificar conteúdo placeholder
            $browser->assertSeeIn('#analyticsModal .modal-body',
                                 'Dados de visualização para esta publicação aparecerão aqui.');
        });
    }

    /**
     * Teste: Ícone de analytics está presente na coluna de visualizações
     */
    public function testIconeAnalyticsEstaPresenteNaColunaVisualizacoes()
    {
        $this->browse(function (Browser $browser) {
            // Criar dados de teste
            $user = $this->criarUsuarioAutenticado();
            $publicacao = $this->criarPublicacao(['total_visualizacoes' => 123]);

            // Fazer login
            $browser->loginAs($user)
                    ->visit('/publicacoes')
                    ->waitFor('table', 5);

            // Verificar que o ícone está presente
            $browser->assertPresent('i.ph-fill.ph-chart-pie-slice')
                    ->assertSee('000123'); // Número formatado com zeros à esquerda
        });
    }

    /**
     * Teste: Dados do modal (data-attributes) estão corretos
     */
    public function testDadosDoModalDataAttributesEstaoCorretos()
    {
        $this->browse(function (Browser $browser) {
            // Criar dados de teste
            $user = $this->criarUsuarioAutenticado();
            $publicacao = $this->criarPublicacao([
                'titulo' => 'Campanha Teste XYZ'
            ]);

            // Fazer login
            $browser->loginAs($user)
                    ->visit('/publicacoes')
                    ->waitFor('table', 5);

            // Verificar atributos data-* do link
            $browser->assertAttribute(
                '[data-publicacao-id="' . $publicacao->id . '"]',
                'data-publicacao-id',
                (string) $publicacao->id
            );

            $browser->assertAttribute(
                '[data-publicacao-id="' . $publicacao->id . '"]',
                'data-publicacao-title',
                'Campanha Teste XYZ'
            );

            $browser->assertAttribute(
                '[data-publicacao-id="' . $publicacao->id . '"]',
                'data-bs-toggle',
                'modal'
            );

            $browser->assertAttribute(
                '[data-publicacao-id="' . $publicacao->id . '"]',
                'data-bs-target',
                '#analyticsModal'
            );
        });
    }

    /**
     * Helper: Criar usuário autenticado para testes
     */
    private function criarUsuarioAutenticado()
    {
        $empresa = Empresa::factory()->create([
            'tipo_empresa' => 'matriz',
            'status' => 1,
        ]);

        return User::factory()->create([
            'empresa_id' => $empresa->id,
            'status_cadastro' => 'aprovado',
            'is_superadmin' => true,
        ]);
    }

    /**
     * Helper: Criar publicação para testes
     */
    private function criarPublicacao(array $attributes = [])
    {
        $franquia = Empresa::factory()->create(['tipo_empresa' => 'franquia']);
        $cliente = Empresa::factory()->create(['tipo_empresa' => 'cliente']);
        $criativo = Criativo::factory()->create(['status_aprovacao' => 'aprovado']);

        return Publicacao::factory()->create(array_merge([
            'franquia_id' => $franquia->id,
            'cliente_id' => $cliente->id,
            'criativo_id' => $criativo->id,
            'titulo' => 'Publicação Teste',
            'status' => 'ativa',
            'tipo_alcance' => 'todos',
            'total_visualizacoes' => 0,
        ], $attributes));
    }
}
