<?php

namespace Tests\Browser;

use App\Models\Publicacao;
use App\Models\AuditoriaSplash;
use App\Models\User;
use Carbon\Carbon;
use Laravel\Dusk\Browser;
use Tests\DuskTestCase;

class PublicacaoEstatisticasTest extends DuskTestCase
{
    protected $user;
    protected $publicacao;

    protected function setUp(): void
    {
        parent::setUp();
        
        $this->user = User::factory()->create([
            'email' => 'test@example.com',
            'password' => bcrypt('password'),
        ]);

        $this->publicacao = Publicacao::factory()->create([
            'data_inicio' => Carbon::now()->subDays(10),
            'data_fim' => Carbon::now()->addDays(10),
            'status' => 'ativa',
        ]);

        // Criar dados de teste
        $this->createTestData();
    }

    /**
     * Teste: Página de estatísticas carrega e exibe corretamente
     */
    public function test_estatisticas_page_loads_and_displays_data()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->user)
                ->visit(route('publicacoes.estatisticas', $this->publicacao))
                ->assertSee('Estatísticas da Publicação')
                ->assertSee($this->publicacao->titulo)
                ->assertSee('Top 10 Hotspots - Mais Impressões')
                ->assertSee('Dados por Dia - Últimos 7 dias')
                ->assertSee('Usuários que Visualizaram a Publicação');
        });
    }

    /**
     * Teste: Cards de estatísticas gerais exibem números
     */
    public function test_general_statistics_cards_display_data()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->user)
                ->visit(route('publicacoes.estatisticas', $this->publicacao))
                ->assertSeeIn('.stat-card', 'Impressões') // Total de impressões
                ->assertSeeIn('.stat-card', 'Cliques')     // Total de cliques
                ->assertSeeIn('.stat-card', 'CTR')         // Taxa de clique
                ->assertSeeIn('.stat-card', 'Usuários');   // Usuários únicos
        });
    }

    /**
     * Teste: Tabela de hotspots exibe dados
     */
    public function test_hotspots_table_displays_data()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->user)
                ->visit(route('publicacoes.estatisticas', $this->publicacao))
                ->assertPresent('table')
                ->assertSeeIn('table', 'Serial')
                ->assertSeeIn('table', 'Impressões')
                ->assertSeeIn('table', 'CTR');
        });
    }

    /**
     * Teste: Tabela de usuários exibe dados com formatação correta
     */
    public function test_users_table_displays_correctly_formatted_data()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->user)
                ->visit(route('publicacoes.estatisticas', $this->publicacao))
                ->scrollTo('h5:contains("Usuários que Visualizaram")')
                ->assertSeeIn('table', 'Data/Hora')
                ->assertSeeIn('table', 'MAC')
                ->assertSeeIn('table', 'Hotspot');
        });
    }

    /**
     * Teste: Breadcrumb navega corretamente
     */
    public function test_breadcrumb_navigation_works()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->user)
                ->visit(route('publicacoes.estatisticas', $this->publicacao))
                ->assertSeeLink('Publicações')
                ->assertSeeLink($this->publicacao->titulo)
                ->click('a:contains("Publicações")')
                ->assertRouteIs('publicacoes.index');
        });
    }

    /**
     * Teste: Botão Voltar funciona
     */
    public function test_back_button_navigation()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->user)
                ->visit(route('publicacoes.estatisticas', $this->publicacao))
                ->assertSee('Voltar')
                ->click('a:contains("Voltar")')
                ->assertRouteIs('publicacoes.show', $this->publicacao);
        });
    }

    /**
     * Teste: Botão Editar funciona
     */
    public function test_edit_button_navigation()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->user)
                ->visit(route('publicacoes.estatisticas', $this->publicacao))
                ->assertSee('Editar')
                ->click('a:contains("Editar")')
                ->assertRouteIs('publicacoes.edit', $this->publicacao);
        });
    }

    /**
     * Teste: Dados de publicação são exibidos no card de informações
     */
    public function test_publication_info_card_displays_correct_data()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->user)
                ->visit(route('publicacoes.estatisticas', $this->publicacao))
                ->assertSee($this->publicacao->titulo)
                ->assertSee($this->publicacao->cliente->name)
                ->assertSee($this->publicacao->franquia->name)
                ->assertSee($this->publicacao->status);
        });
    }

    /**
     * Teste: Usuário não autenticado é redirecionado para login
     */
    public function test_unauthenticated_user_redirects_to_login()
    {
        $this->browse(function (Browser $browser) {
            $browser->visit(route('publicacoes.estatisticas', $this->publicacao))
                ->assertRouteIs('login');
        });
    }

    /**
     * Teste: MAC address é formatado com privacidade
     */
    public function test_mac_address_is_formatted_for_privacy()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->user)
                ->visit(route('publicacoes.estatisticas', $this->publicacao))
                ->scrollTo('h5:contains("Usuários que Visualizaram")')
                // MAC deve estar parcialmente mascarado (***XXXX pattern)
                ->assertSeeIn('table', '***');
        });
    }

    /**
     * Teste: Datas estão formatadas corretamente (d/m/Y)
     */
    public function test_dates_are_formatted_correctly()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->user)
                ->visit(route('publicacoes.estatisticas', $this->publicacao))
                // Data deve estar no formato brasileiro (dd/mm/yyyy)
                ->assertSeeIn('.card-body', $this->publicacao->data_inicio->format('d/m/Y'));
        });
    }

    /**
     * Teste: Página responsiva em mobile
     */
    public function test_page_is_responsive_on_mobile()
    {
        $this->browse(function (Browser $browser) {
            $browser->resize(375, 812) // iPhone dimensions
                ->loginAs($this->user)
                ->visit(route('publicacoes.estatisticas', $this->publicacao))
                ->assertSee('Estatísticas da Publicação')
                ->assertPresent('.table-responsive'); // Tabela deve ter wrapper responsivo
        });
    }

    /**
     * Teste: Gráfico de dados por dia está presente
     */
    public function test_daily_data_chart_is_present()
    {
        $this->browse(function (Browser $browser) {
            $browser->loginAs($this->user)
                ->visit(route('publicacoes.estatisticas', $this->publicacao))
                ->assertSeeIn('h6', 'Dados por Dia - Últimos 7 dias')
                ->assertPresent('canvas, svg'); // Gráfico deve estar presente
        });
    }

    /**
     * Helper: Criar dados de teste
     */
    private function createTestData()
    {
        // Criar registros em diferentes hotspots
        for ($i = 1; $i <= 5; $i++) {
            for ($j = 1; $j <= $i * 2; $j++) {
                AuditoriaSplash::create([
                    'hs_serial' => "HS{$i}",
                    'cl_mac_address' => 'AA:BB:CC:DD:EE:' . str_pad($j, 2, '0', STR_PAD_LEFT),
                    'pag_click' => 'http://example.com/page-' . $j,
                    'imagem' => 'criativo_' . $i . '.jpg',
                    'dsc_identity' => 'Equipment-' . $i,
                    'created_at' => now()->subDays(rand(0, 9))->setHour(rand(8, 18)),
                ]);
            }
        }
    }
}
