<?php

namespace Tests\Browser;

use App\Models\User;
use App\Models\Empresa;
use App\Models\Publicacao;
use App\Models\Criativo;
use Illuminate\Foundation\Testing\DatabaseMigrations;
use Laravel\Dusk\Browser;
use Tests\DuskTestCase;

class PublicacaoPaginacaoComDadosTest extends DuskTestCase
{
    use DatabaseMigrations;

    /**
     * Teste: Paginação aparece quando há mais de 15 registros
     */
    public function testPaginacaoApareceComMaisDe15Registros()
    {
        $this->browse(function (Browser $browser) {
            // Criar usuário autenticado
            $user = $this->criarUsuarioAutenticado();

            // Criar 20 publicações para forçar paginação (padrão é 15 por página)
            for ($i = 1; $i <= 20; $i++) {
                $this->criarPublicacao([
                    'titulo' => "Publicação Teste {$i}"
                ]);
            }

            // Fazer login
            $browser->loginAs($user)
                    ->visit('/publicacoes')
                    ->waitFor('table', 10)
                    ->assertSee('Lista de Publicações');

            // Capturar screenshot
            $browser->screenshot('paginacao-com-dados');

            // Verificar se elemento de paginação existe
            $paginacaoExiste = $browser->script("return !!document.querySelector('.pagination');");
            dump('Paginação existe:', $paginacaoExiste);

            // Verificar se há elemento de paginação
            $browser->assertPresent('.pagination')
                    ->assertSee('Publicação Teste 1')
                    ->assertSee('Publicação Teste 15')
                    ->assertDontSee('Publicação Teste 16'); // Não deve ver a 16ª (página 2)

            // Verificar HTML da paginação
            $paginacaoHtml = $browser->script("return document.querySelector('.pagination')?.outerHTML;");
            dump('HTML da Paginação:', $paginacaoHtml);

            // Verificar classes Bootstrap 5
            $classes = $browser->script("
                const pag = document.querySelector('.pagination');
                if (pag) {
                    return {
                        paginationClasses: pag.className,
                        hasPageItems: pag.querySelectorAll('.page-item').length > 0,
                        hasPageLinks: pag.querySelectorAll('.page-link').length > 0
                    };
                }
                return null;
            ");
            dump('Classes da paginação:', $classes);

            // Clicar no botão "Próximo" se existir
            if ($browser->element('.page-link[rel="next"]')) {
                $browser->screenshot('antes-proximo-pagina');

                $browser->click('.page-link[rel="next"]')
                        ->pause(2000)
                        ->screenshot('depois-proximo-pagina');

                // Na página 2, deve ver da 16 a 20
                $browser->assertSee('Publicação Teste 16')
                        ->assertSee('Publicação Teste 20')
                        ->assertDontSee('Publicação Teste 1'); // Não deve ver a 1ª (página 1)

                dump('Navegação para próxima página funcionou!');
            }

            // Verificar estilos aplicados
            $estilos = $browser->script("
                const pagination = document.querySelector('.pagination');
                if (pagination) {
                    const computed = window.getComputedStyle(pagination);
                    return {
                        display: computed.display,
                        visibility: computed.visibility,
                        opacity: computed.opacity,
                        color: computed.color
                    };
                }
                return null;
            ");
            dump('Estilos CSS aplicados:', $estilos);
        });
    }

    /**
     * Teste: Verificar estrutura correta do Bootstrap 5
     */
    public function testEstruturaPaginacaoBootstrap5()
    {
        $this->browse(function (Browser $browser) {
            // Criar usuário e dados
            $user = $this->criarUsuarioAutenticado();

            for ($i = 1; $i <= 20; $i++) {
                $this->criarPublicacao(['titulo' => "Teste {$i}"]);
            }

            // Login e navegar
            $browser->loginAs($user)
                    ->visit('/publicacoes')
                    ->waitFor('table', 10);

            // Verificar estrutura Bootstrap 5
            $estrutura = $browser->script("
                const pag = document.querySelector('.pagination');
                if (!pag) return { error: 'Paginação não encontrada' };

                return {
                    temClassePagination: pag.classList.contains('pagination'),
                    totalPageItems: pag.querySelectorAll('.page-item').length,
                    totalPageLinks: pag.querySelectorAll('.page-link').length,
                    temItemAtivo: pag.querySelectorAll('.page-item.active').length > 0,
                    temItemDisabled: pag.querySelectorAll('.page-item.disabled').length > 0,
                    primeiroItemTexto: pag.querySelector('.page-item')?.textContent.trim(),
                    ultimoItemTexto: Array.from(pag.querySelectorAll('.page-item')).pop()?.textContent.trim()
                };
            ");

            dump('Estrutura Bootstrap 5:', $estrutura);

            // Assertions
            $browser->assertPresent('.pagination')
                    ->assertPresent('.page-item')
                    ->assertPresent('.page-link')
                    ->assertPresent('.page-item.active');

            $browser->screenshot('estrutura-bootstrap5');
        });
    }

    /**
     * Teste: Informações de paginação ("Showing X to Y of Z results")
     */
    public function testInformacoesPaginacao()
    {
        $this->browse(function (Browser $browser) {
            // Criar dados
            $user = $this->criarUsuarioAutenticado();

            for ($i = 1; $i <= 25; $i++) {
                $this->criarPublicacao(['titulo' => "Item {$i}"]);
            }

            // Login
            $browser->loginAs($user)
                    ->visit('/publicacoes')
                    ->waitFor('table', 10);

            // Verificar texto de informação (Showing 1 to 15 of 25 results)
            $textoInfo = $browser->script("
                // Bootstrap 5 pagination mostra essas informações
                const nav = document.querySelector('nav');
                if (nav) {
                    const paragrafos = nav.querySelectorAll('p');
                    return Array.from(paragrafos).map(p => p.textContent.trim());
                }
                return [];
            ");

            dump('Informações de paginação:', $textoInfo);

            $browser->screenshot('info-paginacao');
        });
    }

    /**
     * Helper: Criar usuário autenticado
     */
    private function criarUsuarioAutenticado()
    {
        $empresa = Empresa::factory()->create([
            'tipo_empresa' => 'matriz',
            'status' => 1,
        ]);

        return User::factory()->create([
            'empresa_id' => $empresa->id,
            'status_cadastro' => 'aprovado',
            'is_superadmin' => true,
        ]);
    }

    /**
     * Helper: Criar publicação
     */
    private function criarPublicacao(array $attributes = [])
    {
        $franquia = Empresa::factory()->create(['tipo_empresa' => 'franquia']);
        $cliente = Empresa::factory()->create(['tipo_empresa' => 'cliente']);
        $criativo = Criativo::factory()->create(['status_aprovacao' => 'aprovado']);

        return Publicacao::factory()->create(array_merge([
            'franquia_id' => $franquia->id,
            'cliente_id' => $cliente->id,
            'criativo_id' => $criativo->id,
            'status' => 'ativa',
            'tipo_alcance' => 'todos',
        ], $attributes));
    }
}
